from django.contrib.auth.models import User
from django.contrib.auth.backends import BaseBackend

from schedule.helpers import arbutus_wc_authenticate

from result_manager.arbutus.connection import arbutus_em_authenticate

from .cryptor import encrypt_password
from .models import Password


class AnalyzerUserAuthBackend(BaseBackend):

    def authenticate(self, request, username, password):
        """

        :param username:
        :param password:
        :return User:
        """

        # this is a django user, as opposed to an arbutus user profile
        # both use the entered username as a 'key' reference.
        user = None

        try:
            is_auth = False
            if request.path == '/result_manager/':
              is_auth = arbutus_em_authenticate(username, password)
            else:       # webconnect
              is_auth = arbutus_wc_authenticate(username, password)

            if is_auth:
                try:
                    # Try to find a user matching your username
                    user = User.objects.get(username__iexact=username)
                except User.DoesNotExist:
                    # No user was found, but we have success on authenticate, create user
                    user = User.objects.create_user(username, password=password)
            else:
                pass

        except AttributeError:
            pass

        return user

    # Required for your backend to work properly - unchanged in most scenarios
    def get_user(self, user_id):
        """

        :param user_id:
        :return:
        """
        try:
            return User.objects.get(pk=user_id)
        except User.DoesNotExist:
            return None


def save_arbutus_passwd(user_name, password):
    """
    keep the username and password (encrypted) in database so that we can call server
    without asking the user for authentication details again while web session is alive.
    should check if user is authenticated/logged in before using
    :param user_name:
    :param password:
    :return:
    """

    try:
        passwd = Password.objects.filter(username__iexact=user_name).first()
        if passwd is None:
            passwd = Password.objects.create(username=user_name)

        # passwd.delete()  # remove previous entry as we only want to store current successful auth.
    except Password.DoesNotExist:
        passwd = Password.objects.create(username=user_name)

    # create new entry with current authentication details
    # passwd = Password()
    # passwd.user = user.username

    # always update password!
    enc_password = encrypt_password(password)
    passwd.password = enc_password.raw
    passwd.save()
