import json
from tkinter.tix import FileSelectBox

from ..arbutus.messages import get_enc_password, get_user_id
from ..arbutus_server.connection import full_server_connect
from ..arbutus_server.file_io import handle_uploaded_file, handle_download_file
from ..views_helper import PAGE_FIELD_NAMES

from .connection import do_server_socket, is_auth_ok
from .do_messages import (
    do_delete_messages, do_em_messages, do_dlg_messages,
    do_dlg_add_or_update_messages,
    do_close_activity_1, do_close_activity_2, do_reassign_activities,
    do_get_notes_message, do_save_notes_message, do_config_messages,
    do_reports, do_restore, do_enum_log, do_add_answer,
)
from .messages import get_server_name


# special process for attachment upload
def process_file_upload(username, password, files, thread_id):
    server_socket, result = do_server_socket(username, password)
    if is_auth_ok(result.get('cmd_type'), result.get('dest_srce')):
        connected, error = full_server_connect(server_socket, username, get_server_name())
        if not connected:
            return False, error
        else:
            attachment_id, who, created = handle_uploaded_file(server_socket, files, username, thread_id)

        # TODO errors?
        return attachment_id, who, created


# another special case for attachment download.
def process_file_download(username, password, attach_id):
    server_socket, result = do_server_socket(username, password)
    if is_auth_ok(result.get('cmd_type'), result.get('dest_srce')):
        connected, error = full_server_connect(server_socket, username, get_server_name())
        if not connected:
            return False, error
        else:
            return handle_download_file(server_socket, attach_id), None


# CRUD - Create, Read, Update, Delete
# process the message or new or change dialogs

def process_messages(user_name, enc_password, do_message, *args):
    # Generic process message handler, the workhorse that takes
    # care of most (see special cases above) arbutusserver
    # authentication and connection needs from view Requests.
    extra_result = None, None
    server_socket, result = do_server_socket(user_name, enc_password)
    return_result = None
    if is_auth_ok(result.get('cmd_type'), result.get('dest_srce')):
        connected, error = full_server_connect(server_socket, user_name, get_server_name())
        if not connected:
            return_result = None, error
        else:
            return_result, extra_result = do_message(server_socket, user_name, *args)

    server_socket.close()
    return return_result, extra_result


def process_em_messages(msg_type, user_name, enc_password, msg_filter):
    return process_messages(user_name, enc_password, do_em_messages, [msg_type, msg_filter])


def process_dlg_messages(dlg_type, user_name, enc_password, list_id):
    return process_messages(user_name, enc_password, do_dlg_messages, [dlg_type, list_id])


def process_dlg_add_or_update_messages(command, dlg_type, user_name, enc_password, user_data):
    return process_messages(user_name, enc_password, do_dlg_add_or_update_messages, [dlg_type, command, user_data])


def process_delete_messages(dlg_type, user_name, enc_password, rows):
    return process_messages(user_name, enc_password, do_delete_messages, [dlg_type, rows])


def process_config_messages(msg_type, user_name, enc_password, grid_config):
    return process_messages(user_name, enc_password, do_config_messages, [msg_type, grid_config])


def process_close_activity(user_name, enc_password, data_id, data):
    if data:
        # data_id is just the user_id which could be obtained from get_user_id on the user_name
        # however, to prevent confusion, we pass it as its own identity as close_activity_1
        # uses the data_id to hold the activity_id and no data
        return process_messages(user_name, enc_password, do_close_activity_2, [data_id, data])
    else:
        # data_is is the activity_id
        return process_messages(user_name, enc_password, do_close_activity_1, [data_id])


def process_reassign_activities(user_name, enc_password, new_user_id, activities):
    return process_messages(user_name, enc_password, do_reassign_activities, [new_user_id, activities])


def process_reports(user_name, enc_password, chart_id, category, period):
    return process_messages(user_name, enc_password, do_reports, [chart_id, category, period])


def process_restore(user_name, enc_password, backup_stamp, log_list):
    return process_messages(user_name, enc_password, do_restore, [backup_stamp, log_list])


def process_enum_log(user_name, enc_password, backup_stamp):
    return process_messages(user_name, enc_password, do_enum_log, [backup_stamp])

def process_close_activity_add_answers(user_name, enc_password, rmform_answers):
    return process_messages(user_name, enc_password, do_add_answer, [rmform_answers])

# process ajax calls

def process_dlg_details(username, request_method, request_params):

    dialog_type = request_params.get('dialog_type')

    if request_method == 'GET':
        list_id = request_params.get('list_id')
        record = {}
        result, extra_result = process_dlg_messages(dialog_type, username, get_enc_password(username), list_id)

        # handle errors on result
        if result[0].get('cmd_type') != 33000:
            error = result[0].get('buffer').decode('utf-8')
        else:
            # convert result to an array of data
            buf_list = result[0].get('buffer').decode('utf-8').split('\t')
            # remove null at end of last elem
            buf_list[-1] = buf_list[-1].rstrip('\x00')
            idx = 0
            for field_name in PAGE_FIELD_NAMES.get(dialog_type):
                # no id returned in users buf
                if dialog_type == 'users' and 'id' not in record:
                    record['id'] = int(list_id)
                    continue

                try:
                    record[field_name] = buf_list[idx]
                except IndexError:
                    # allows us to skip less buffer field values than page field names,
                    # assuming order matches what is returned in buffer
                    # with the order specified in PAGE_FIELD_NAMES
                    pass

                idx += 1

            # additional processing for users and groups for member list
            error = None
            if dialog_type == 'users':
                record['groups'], error = extract_extra_result(extra_result)
            elif dialog_type == 'groups':
                record['users'], error = extract_extra_result(extra_result)

        result = {'error': error, 'result': record}
        json_data = json.dumps(result)

    else:   # 'POST'
        command = request_params.get("command")
        rows = []
        dlg_data = {}

        if command == 'Delete':
            rows = json.loads(request_params.get("delete_rows"))
        else:
            dlg_data = json.loads(request_params.get("form_data"))

        if command == 'Delete':
            result, _ = process_delete_messages(dialog_type, username, get_enc_password(username), rows)
        else:
            result, _ = process_dlg_add_or_update_messages(command, dialog_type, username, get_enc_password(username), dlg_data)

        status, message, data = check_result(result, command, dialog_type, dlg_data)
        output_result = { 'result': status, 'message': message, 'data': data }
        json_data = json.dumps(output_result)

    return json_data


def prepare_process_close_activity(username, request_method, request_params):
    if request_method == 'GET':
        activity_id = int(request_params.get('activity_id'))
        enum_activity_result, enum_extra_results = process_close_activity(
            username, get_enc_password(username), activity_id, None
        )
        output_result = {
            'enum_activity_success': enum_activity_result[0],
            'enum_activity_error': enum_activity_result[1],
            'enum_action_success': enum_extra_results[0],
            'enum_action_error': enum_extra_results[1],
            'enum_attach_success': enum_extra_results[2],
            'enum_attach_error': enum_extra_results[3],
            'enum_notes_success': enum_extra_results[4],
            'enum_notes_error': enum_extra_results[5]
        }
    else:  # 'POST'
        result, _ = process_close_activity(
            username, get_enc_password(username),
            get_user_id(username),
            json.loads(request_params.get("form_data"))
        )

        output_result = {'result': result.get('result', ''), 'message': result.get('message', ''), 'data': ''}

    return json.dumps(output_result)


def process_notes(user_name, enc_password, thread_id, new_note):
    result = None
    if new_note is None:
        result, _ = process_messages(user_name, enc_password, do_get_notes_message, [thread_id])
    else:
        result, _ = process_messages(user_name, enc_password, do_save_notes_message, [thread_id, new_note])

    return result


# private helpers

def extract_extra_result(result):
    records = []
    if result[0] is not None:
        if result[0].get('cmd_type') == 33000:

            buf_list = result[0].get('buffer').decode('utf-8').split('\x00')
            # remove null at end of last elem
            buf_list = buf_list[:-2]
            for item in buf_list:
                if len(item) > 0:
                    records.append(item.split('\t')[0])

            return records, None
        else:
            return None, result[0].get('buffer').decode('utf-8')
    else:
        return None, None


def check_result(result, command, dlg_type, dlg_data):
    if result.get('cmd_type') == 33000:
        status = 'OK'
        message = ''
        data = {}
        if command == 'Create2' or command == 'Create3':
            # TODO
            data['id'] = result.get('dest_srce', None)
            data['name'] = dlg_data.get('name', None)
        elif command == 'Create' and dlg_type == 'visual-forms':
            message = dlg_type
            data['rmf_id'] = result.get('dest_srce', None)
            data['node_id'] = dlg_data.get('node_id', None)
        elif command == 'Create' and dlg_type == 'threads':
            message = dlg_type
            data['thread_id'] = result.get('dest_srce', None)
        elif command != 'Delete' and dlg_type == 'processes':
            # processes dialog results handled differently
            # as we redirect to the visual editor after saving their values.
            message = 'process_redirect'
            data['process_id'] = dlg_data.get("process_id")
            data['first_state_id'] = dlg_data.get("first_state_id", 0)
            data['process_label'] = dlg_data.get('label')
            data['process_description'] = dlg_data.get('description')
            data['process_serialno'] = dlg_data.get('serialno')
            data['process_entry'] = dlg_data.get('entry')
    else:
        status = 'FAIL'
        message = result.get('buffer').decode('utf-8')
        data = None

    return status, message, data


def process_backup_result(result, msg_type):

    try:
        result_buffer = result.get('buffer')
    except AttributeError:
        result_buffer = result[0].get('buffer')

    if len(result_buffer) > 0:
        result_buffer = result_buffer.decode('utf-8').split('\x00')[0]

    result_output = ''
    error_output = ''
    try:
        if result[0].get('cmd_type') == 33000:
            result_output = result_buffer
        else:
            error_output = result_buffer
    except Exception as e:
        if result.get('cmd_type') == 33000:
            result_output = 'Success'
        else:
            error_output = result_buffer

    return {'msg_type': msg_type, 'result': result_output, 'em_error': error_output}
