import json

from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.models import User
from django.contrib.auth.mixins import LoginRequiredMixin

from django.views.decorators.http import require_POST
from django.views.decorators.csrf import csrf_exempt
from django.shortcuts import render, redirect
from django.views.generic import FormView
from django.http import HttpResponse, StreamingHttpResponse

from common.auth import save_arbutus_passwd

from .forms import LoginForm
from .views_helper import PAGE_FIELD_NAMES, extract_em_messages_result

from .arbutus.messages import get_enc_password, is_em_admin, get_name
from .arbutus.process_messages import (process_em_messages, 
  process_file_upload, process_file_download
)

from .arbutus_server.file_io import get_file_download, build_asi_packet_file_read


########################################################################################################


class LoginView(FormView):
    template_name = 'result_manager/index.html'
    form_class = LoginForm
    success_url = 'dashboard'

    def get(self, request, *args, **kwargs):
        form = self.form_class()

        return render(
            request,
            self.template_name,
            {'form': form, 'logo': None, }
        )

    def form_valid(self, form):

        username = form.cleaned_data['username']
        password = form.cleaned_data['password']

        # remove windows logon user from django user table before each auth.
        try:
            User.objects.get(username__iexact=username, is_staff=False, is_superuser=False).delete()
        except User.DoesNotExist:
            pass
        
        user = authenticate(
            self.request,
            username=username,
            password=password
            #arbutus_authenticate=arbutus_em_authenticate
        )
            

         # do not reserved user 'admin' as it is not a windows login authenticated user
        if user is not None and user.is_active and user.is_superuser != True:
            save_arbutus_passwd(username, password)
            logout(self.request)
            login(self.request, user)

            # send over EM messages to arbutus server
            # TODO review this special message
            # that persists an em_login user details
            # on success.
            # TODO show form error as appropriate
            # remember that a user can be a valid windows login
            # BUT not be a user of EM.
            result, _ = process_em_messages(
                'emlogin',
                username,
                get_enc_password(username),
                [None, 0]
            )

            try:
                if len(result) == 2 and result[1] != None and result[1].get('cmd_type') != 33000:
                    form.add_error(None, result[1].get('buffer').decode('utf-8'))
                    return self.form_invalid(form)
                elif result[0] != None and result[0].get('cmd_type') != 33000:
                    form.add_error(None, result[0].get('buffer').decode('utf-8'))
                    return self.form_invalid(form)
            except TypeError:
                form.add_error('password', 'Please correct EM server profile')
                return self.form_invalid(form)

            if not is_em_admin(self.request.user.username):
                self.success_url = 'activities'

            return super(LoginView, self).form_valid(form)
        else:
            form.add_error('username', '')
            form.add_error('password', 'Credentials incorrect! Try again.')
            return self.form_invalid(form)


class LogoutView(LoginRequiredMixin, FormView):

    def get(self, request, *args, **kwargs):
        logout(request)
        return redirect('em-login')


class ActivityView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/activities.html'

    def get(self, request, *args, **kwargs):
        # form = self.form_class()  # initial=self.initial)

        thread_id = request.GET.get('thread', None)

        if request.user.is_authenticated:

            # users list required for reassign dialog.
            result, _ = process_em_messages(
                'users',
                request.user.username,
                get_enc_password(request.user.username),
                [None, 0]
            )
           
            # prune result into just ids and corresponding group names
            return render(request, self.template_name, {
                'is_em_admin': is_em_admin(request.user.username),
                'user_name': get_name(request.user.username),
                'users': sorted(extract_em_messages_result(
                    result,
                    PAGE_FIELD_NAMES['users'][:2]),
                    key=lambda k: k['name'].lower()
                ),
                'thread_activities_id': thread_id
            })

        return render(request, self.template_name, {
            'is_em_admin': is_em_admin(request.user.username),
            'thread_activities_id': thread_id
        })


class UsersView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/users.html'

    def get(self, request, *args, **kwargs):
        is_admin = is_em_admin(request.user.username)
        if not is_admin:
            self.template_name = 'result_manager/forbidden.html'

        result, _ = process_em_messages(
            'groups',
            request.user.username,
            get_enc_password(request.user.username),
            [None, 0]
        )

        # prune result into just ids and corresponding group names
        return render(request, self.template_name, {
            'is_em_admin': is_admin,
            'user_name': get_name(request.user.username),
            'groups': sorted(extract_em_messages_result(
                result,
                PAGE_FIELD_NAMES['groups'][:2]),
                key=lambda k: k['name'].lower()
            )
        })


class GroupsView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/groups.html'

    def get(self, request, *args, **kwargs):
        is_admin = is_em_admin(request.user.username)
        if not is_admin:
            self.template_name = 'result_manager/forbidden.html'

        result, _ = process_em_messages(
            'users',
            request.user.username,
            get_enc_password(request.user.username),
            [None, 0]
        )

        # prune result into just ids and corresponding group names
        return render(request, self.template_name, {
            'is_em_admin': is_admin,
            'user_name': get_name(request.user.username),
            'users': sorted(extract_em_messages_result(
                result,
                PAGE_FIELD_NAMES['users'][:2]),
                key=lambda k: k['name'].lower()
            ),
        })


class ProcessesView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/processes.html'

    def get(self, request, *args, **kwargs):
        is_admin = is_em_admin(request.user.username)
        if not is_admin:
            self.template_name = 'result_manager/forbidden.html'

        return render(request, self.template_name, {
            'is_em_admin': is_admin,
            'user_name': get_name(request.user.username)
        })


class ThreadsView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/threads.html'

    def get(self, request, *args, **kwargs):
        # threads is not an admin only option, as no admin id is required
        # to add them etc.
        # if not is_admin:
        #    self.template_name = 'result_manager/forbidden.html'
        result, _ = process_em_messages(
            'processes',
            request.user.username,
            get_enc_password(request.user.username),
            [None, 0]
        )

        # prune result into just ids and corresponding process names from result & sort by process name
        processes = sorted(extract_em_messages_result(result, PAGE_FIELD_NAMES['processes'][:5]),
            key=lambda k: k['label'].lower()
        )

        # escape any hashes (#) in processes name to appease client-side template rendering
        escaped_processes = []
        for process in processes:
            escaped_process_name = process['label'].replace("#", "\\#")
            escaped_processes.append( {'id': process['id'], 'label': escaped_process_name, 'entry': process['entry']} )

        return render(request, self.template_name, {
            'is_em_admin': is_em_admin(request.user.username),
            'user_name': get_name(request.user.username),
            'user': request.user.username,
            'processes': escaped_processes
        })


class DashboardView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/dashboard.html'

    def get(self, request, *args, **kwargs):
        is_admin = is_em_admin(request.user.username)
        if not is_admin:
            self.template_name = 'result_manager/forbidden.html'

        return render(request, self.template_name, {
            'is_em_admin': is_em_admin(request.user.username),
            'user_name': get_name(request.user.username)
        })


class AdminView(LoginRequiredMixin, FormView):
    template_name = 'result_manager/admin.html'

    def get(self, request, *args, **kwargs):
        is_admin = is_em_admin(request.user.username)
        if not is_admin:
            self.template_name = 'result_manager/forbidden.html'

        return render(request, self.template_name, {
            'is_em_admin': is_admin,
            'user_name': get_name(request.user.username)
        })


# visual editor view

class VisualEditorView(FormView):
    template_name = 'result_manager/veditor.html'

    def get(self, request, *args, **kwargs):
        result_users, _ = process_em_messages(
            'users',
            request.user.username,
            get_enc_password(request.user.username),
            [None, 0]
        )

        result_groups, _ = process_em_messages(
            'groups',
            request.user.username,
            get_enc_password(request.user.username),
            [None, 0]
        )

        users = [dict(item, **{'type': 'u'})
                 for item in extract_em_messages_result(result_users, PAGE_FIELD_NAMES['users'][:2])]
        groups = [dict(item, **{'type': 'g'})
                  for item in extract_em_messages_result(result_groups, PAGE_FIELD_NAMES['groups'][:2])]

        return render(request, self.template_name, {
            'is_em_admin': is_em_admin(request.user.username),
            'user_name': get_name(request.user.username),
            'owners':  sorted(users + groups, key=lambda k: k['name'].lower())
        })


# attachment upload/download handlers

@csrf_exempt
@require_POST
def upload(request):

    # The assumption here is that jQuery File Upload
    # has been configured to send files one at a time.
    # If multiple files can be uploaded simultaneously,
    # 'file' may be a list of files.
    file = request.FILES['files[]'] if request.FILES else None
    thread_id = request.POST.get('thread_id')
    username = request.user.username
    password = get_enc_password(request.user.username)

    # NOTE: this is now pretty inefficient, as it will be called for every attachment
    # which in turn will now also additional enum all the attachments to find it in the list.
    # this is just so we can get the who and created for a newly attached file.
    attachment_id, who, created = process_file_upload(username, password, file, thread_id)
    url = '/result_manager/attachments?id='
    url += str(attachment_id)

    file_dict = {
        'name': file.name,
        'size': file.size,
        'url': url,
        'who': who,
        'created': created,
    }

    files = file_dict if isinstance( file_dict, list ) else [ file_dict ]
    data = { 'files' : files }
    return HttpResponse(json.dumps(data), content_type="application/json")


def download_file(request):
    if request.method == 'POST':
        pass
    else:
        username = request.user.username
        password = get_enc_password(request.user.username)
        attach_id = request.GET.get('id')
        result, error = process_file_download(username, password, attach_id)
        # TODO handle error
        server_socket = result[0]
        server_id = result[1]
        filename = result[2]
        download_packet = build_asi_packet_file_read(server_id)
        response = StreamingHttpResponse(
            get_file_download(server_socket, server_id, download_packet),
            content_type="application/force-download"
        )
        response['Content-Disposition'] = 'attachment; filename="'+ filename.decode('utf-8') + '"'
        return response
