import json
from html import escape
from collections import OrderedDict

from django.http import JsonResponse

from .views_helper import (
    PAGE_FIELD_NAMES, extract_em_messages_result, 
    change_id_from_number_to_string_value, 
)

from .arbutus.messages import get_enc_password
from .arbutus.process_messages import (
    process_em_messages, process_dlg_details, prepare_process_close_activity,
    process_notes, process_reassign_activities,
    process_config_messages, process_reports, process_backup_result, 
    process_restore, process_enum_log, process_close_activity_add_answers,
)
from .arbutus.process_ve_messages import process_dlg_ve_details


##########################################################################

# JSON API - populate bootgrid/kendoUI grid when view loaded.

##########################################################################

def api_data_basic(request):

    # calculate current grid data to render

    row_count = request.POST.get('pageSize')
    current_page = request.POST.get('page')
    path_page = request.POST.get('pathname').split('/')[-1]
    msg_filter = request.POST.get('msg_filter')  # USER, ACTIVE, ALL
    search_phrase = request.POST.get('searchPhrase')

    # special case to handle redirect from threads to activities listing
    # that just shows a selected threads activities.
    thread_activities = request.POST.get('thread_activities_id')
    thread_activities_id = 0
    if thread_activities:
        thread_activities_id = int(thread_activities)

    # TODO future may hold arbutus server side filters
    # data_filter = request.POST.get('data_filter')

    # calculate paging values etc.
    try:
        start = (int(current_page) - 1) * int(row_count)
        end = start + int(row_count)
    except TypeError:
        current_page = 1
        row_count = 250
        start = 0
        end = 250

    total = 0
    result = []

    if request.user.is_authenticated:
        # use dummy data and base total on length of this data
        # it chooses the appropriate data based on the page.
        # send over EM messages to the server
        result, _ = process_em_messages(
            path_page,
            request.user.username,
            get_enc_password(request.user.username),
            [msg_filter, thread_activities_id]
        )

        result = extract_em_messages_result(result, PAGE_FIELD_NAMES.get(path_page))

        # if sort info, sort on it
        sort_dir = request.POST.get('sort[0][dir]', '')
        sort_key = request.POST.get('sort[0][field]', '')
        #if len(sort_dir) > 0 and len(sort_key) > 0:
        if sort_dir and sort_key:
            is_reverse = False
            if sort_dir == 'desc':
                is_reverse = True

            if any(sort_key in d for d in result):
                result = sorted(result, key=lambda k: k[sort_key], reverse=is_reverse)

        # are we filtering by searchPhrase ?
        if search_phrase:
            search_result = []
            for obj in result:
                row = ''
                # concat each obj values to string
                for key, value in obj.items():
                    if key != 'id':
                        row += value

                # search in string row.
                if search_phrase in row:
                    search_result.append(obj)

            result = search_result

        # total number of records in result
        total = len(result)

        # NOTE: make sure all shows everything

        # handle msg_filter result changing the current page
        if start > total:
            current_page = 1
        elif row_count != total:
            result = result[start:end]

    data = {"total": total, "page": current_page, "pageSize": row_count, "rows": result}
    return JsonResponse(data)


# grid view server side events and dialogs  etc.

def api_data_details(request):
    if request.method == 'GET':
        request_params = request.GET
    else:
        request_params = request.POST

    result = process_dlg_details(request.user.username, request.method, request_params)
    json_data = json.loads(result)
    return JsonResponse(json_data)


def api_data_ve_details(request):
    if request.method == 'GET':
        request_params = request.GET
    else:
        request_params = request.POST

    result = process_dlg_ve_details(request.user.username, request.method, request_params)
    json_data = json.loads(result)
    return JsonResponse(json_data)


def api_data_close_activity(request):
    if request.method == 'GET':
        request_params = request.GET
    else:
        request_params = request.POST

    result = prepare_process_close_activity(request.user.username, request.method, request_params)
    json_data = json.loads(result)
    return JsonResponse(json_data)


def api_data_reassign_activities(request):
    new_user_id = request.POST.get("new_user_id")
    activities = json.loads(request.POST.get("reassign_activities"))
    result = process_reassign_activities(
        request.user.username, get_enc_password(request.user.username), new_user_id, activities
    )
    json_data = result[0]
    return JsonResponse(json_data)


def api_data_notes(request):
    if request.method == 'GET':
        thread_id = request.GET.get('thread_id')
        result = {'all_notes': process_notes(
            request.user.username, get_enc_password(request.user.username), thread_id, None)}
    else:
        thread_id = request.POST.get('thread_id')
        new_note = escape(request.POST.get('new_note').strip('"'))
        result = {'result': process_notes(
            request.user.username, get_enc_password(request.user.username), thread_id, new_note)}

    json_data = result
    return JsonResponse(json_data)


def api_data_grids_config(request):

    grids = None
    if request.method == 'GET':
        msg_type = 'load'
    else:
        grids = request.POST.get('grids')
        if grids == '':
            msg_type = 'clear'
        else:
            msg_type = 'save'
    
    result, _ = process_config_messages(
        msg_type,
        request.user.username,
        get_enc_password(request.user.username),
        grids
    )

    return JsonResponse(result)


def api_data_dbcharts(request, chart_id, category):

    result, _ = process_reports(
        request.user.username,
        get_enc_password(request.user.username),
        int(chart_id),
        category,
        request.GET.get('period', '')
    )

    if int(chart_id) == 5 or int(chart_id) == 8:
        return JsonResponse(result, safe=False)

    json_data = {'data': result}
    return JsonResponse(json_data, safe=False)


def api_data_backup(request):

    msg_type = 'enum-backups'
    if request.method == 'POST':
        msg_type = 'backup'

    result, _ = process_em_messages(
        msg_type,
        request.user.username,
        get_enc_password(request.user.username),
        [None, 0]
    )

    json_output = process_backup_result(result, msg_type)
    return JsonResponse(json_output, safe=False)


def api_data_restore(request):
    log_list = request.POST.get('em_log_selection', None)
    restore_result, log_result = process_restore(
        request.user.username,
        get_enc_password(request.user.username),
        request.POST.get('em_restore_selection', None),
        json.loads(log_list)
    )

    json_output = process_backup_result(restore_result, 'restore')
    # modify return results if log restore point fails
    if log_result is not None and log_result.get('cmd_type') == 33001:
        json_output['result'] = ''
        if len(log_result.get('buffer')) > 0:
            result_buffer = log_result.get('buffer').decode('utf-8').split('\x00')[0]
            json_output['em_error'] = result_buffer
        else:
            json_output['em_error'] = 'Fail'

    return JsonResponse(json_output, safe=False)


def api_data_enum_log(request):

    result, buffer = process_enum_log(
        request.user.username,
        get_enc_password(request.user.username),
        request.POST.get('em_restore_selection')
    )

    error_output = ''
    result_output = ''
    if result.get('cmd_type') == 33000:
        result_output = buffer.decode('utf-8').split('\x00')[:-2]  # remove the double null on the list split
    else:
        error_output = buffer.decode('utf-8').split('\x00')[0]  # if error, get its string from buffer

    json_output = {'msg_type': 'enum-log', 'result': result_output, 'em_error': error_output}
    return JsonResponse(json_output, safe=False)


# visual editor JSON API

# def api_veditor_load(request):
#    return JsonResponse('')


def api_veditor_save(request):
    saved_model_str = request.POST.get("savedModel")
    json_objects = json.loads(saved_model_str)  # , object_pairs_hook=OrderedDict)
    # TODO, should we depend on client side storage across views?
    #label = request.POST.get("process_label")
    #description = request.POST.get("process_description")
    #serialno = request.POST.get("process_serialno")
    #command = request.POST.get("command")
    #if command == 'Update':
    #process_id = int(request.POST.get("process_id"))
    #else:
    #    process_id = 0

    # dictate order of various object keys so that we can ensure that
    # 1. nodes -> links -> ...
    #   2. id -> .... (consistent order of other keys)

    object_keyorder = {k: v for v, k in enumerate(
        ['nodeDataArray', 'linkDataArray', 'class', 'nodeKeyProperty', 'linkKeyProperty', 'copiesArrays',
         'copiesArrayObjects', 'linkFromPortIdProperty', 'linkToPortIdProperty']
    )}
    ordered_objects = OrderedDict(sorted(json_objects.items(), key=lambda i: object_keyorder.get(i[0])))

    new_nodes = []
    node_keyorder = {k: v for v, k in enumerate(
        ['id', 'text', 'desc', 'duration', 'method', 'overdue', 'owner', 'reminder', 'return', 'type',
         'emergency', 'subject', 'body', 'category', 'rmf_id', 'rmf_text', 'rmf_desc', 'rmf_questions',
         'loc', 'name', 'leftArray', 'rightArray', 'topArray', 'bottomArray']
    )}
    for node in ordered_objects['nodeDataArray']:
        change_id_from_number_to_string_value(node)
        new_nodes.append(OrderedDict(sorted(node.items(), key=lambda i: node_keyorder.get(i[0]))))

    ordered_objects['nodeDataArray'] = new_nodes

    new_links = []
    link_keyorder = {k: v for v, k in enumerate(
        ['id', 'to', 'from', 'text', 'choose', 'desc', 'strokeWidth', 'points', 'fromPort', 'toPort']
    )}
    for link in ordered_objects['linkDataArray']:
        change_id_from_number_to_string_value(link)
        new_links.append(OrderedDict(sorted(link.items(), key=lambda i: link_keyorder.get(i[0]))))

    ordered_objects['linkDataArray'] = new_links

    # do not need the class item
    del ordered_objects['class']

    #result = old_process_em_ve_messages(
    #    request.user.username,
    #    get_enc_password(request.user.username),
    #    command,
    #    process_id,
    #    label,
    #    description,
    #    json.dumps(ordered_objects, separators=(',', ':'))
    #)
    
    # look for arbutusserver error
    err = ''
    if result.get('cmd_type') == 33000:
        result = {'result': 'OK'}
    else:
        err = result.get('buffer').decode('utf-8').rstrip('\0')
        result = {'result': 'ERROR', 'error': err}

    return JsonResponse(result)


def api_request_id(request, num_ids):

  rm_error = '' # 'This is an error in the buffer.'
  result, _ = process_em_messages(
      'request-id',
      request.user.username,
      get_enc_password(request.user.username),
      [None, num_ids]
  )
  
  # TODO handle more than one id in buffer
  rm_id = 0
  try:
    buffer = result[0].get('buffer').decode('utf-8').split('\00')[0].split('\t')
    rm_id = buffer[0]
  except AttributeError:
    rm_id = int(result[0].get('dest_srce', 0))
  
  if rm_id == 0:
    rm_error = result[0].get('buffer', '').decode('utf-8').rstrip('\0')

  json_output = {'rm_id': rm_id, 'rm_error': rm_error}
  return JsonResponse(json_output, safe=False)


def api_get_rmforms(request):
  
  search_phrase = json.loads(request.POST.get('search_phrase'))

  rm_error = '' # 'This is an error in the buffer.'
  result, _ = process_em_messages(
      'get-rmforms',
      request.user.username,
      get_enc_password(request.user.username),
      [None, 0]
  )
  
  buf = result[0].get('buffer', '').decode('utf-8').rstrip('\0')
  forms_result = []
  if result[0].get('cmd_type') != 33000:
    rm_error = buf
  else: # convert into JSON list
    forms_result = buf.split('\0')
  
  # filter all forms returned by search_phrase ?
  # the searched string is a concat. of form name & instructions
  if search_phrase:
    search_result = []
    for obj in forms_result:
        row = ''
        # concat each obj values to string
        current_row = obj.split('\t')
        index = 0
        for field in current_row:
            if index == 5:
                row = field
                row += ' '
            elif index == 6:
                row += field
                
            index += 1

        # search in string row.
        if row.lower().find(search_phrase.lower()) > -1:
            search_result.append(obj)

    forms_result = search_result

  json_output = {'forms': forms_result, 'error': rm_error}
  return JsonResponse(json_output, safe=False)


def api_workflow_rollback(request):
  rm_error = '' # 'This is an error in the buffer.'
  process_id = request.POST.get('pid', '')
  result, _ = process_em_messages(
      'workflow-rollback',
      request.user.username,
      get_enc_password(request.user.username),
      [None, process_id]
  )
  
  if result.get('cmd_type') != 33000:
    rm_error = result.get('buffer', '').decode('utf-8').rstrip('\0')

  json_output = {'error': rm_error}
  return JsonResponse(json_output, safe=False)


def api_close_activity_add_answers(request):

    result, _ = process_close_activity_add_answers(
        request.user.username,
        get_enc_password(request.user.username),
        json.loads(request.POST.get('form_data'))
    )

    rm_error = ''
    if result.get('cmd_type') != 33000:
        rm_error = result.get('buffer', '').decode('utf-8').rstrip('\0')

    json_output = {'error': rm_error}
    return JsonResponse(json_output, safe=False)
