import re

from django.contrib import admin
from django.contrib.auth.models import User
from django.contrib.auth.models import Group
from django import forms

from .models import (
    SharedFolder, UserProfile, VariableDescription,
    ServerProfile, GlobalConfig, ODBCProfile,
    filter_user_var_fields,
)

from .forms import ODBCProfileForm


# Kurt has requested that we override the default alphabetical model ordering
# in the Admin protal overview with a crafted model order.
def custom_get_app_list(self, request, app_label=None):
     
    ordering = {
        "SharedFolder": 5,
        "UserProfile": 2,
        "VariableDescription": 6,
        "ServerProfile": 3,
        "GlobalConfig": 1,
        "ODBCProfile": 4,
    }

    app_dict = self._build_app_dict(request)
    # Sort the apps alphabetically.
    app_list = sorted(app_dict.values(), key=lambda x: x['name'].lower())

    # Sort the models alphabetically within each app.
    for app in app_list:
        app['models'].sort(key=lambda x: ordering[x['object_name']])

    return app_list

admin.site.unregister(User)
admin.site.unregister(Group)
admin.AdminSite.get_app_list = custom_get_app_list
admin.site.site_header = 'Arbutus web applications administration'


class SharedFolderAdmin(admin.ModelAdmin):
    class Media:
        js = ('schedule/arbutus_admin_schedule.js', )


    exclude = ('project_name',)

    # TODO review in documentation on use cases.
    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "server_profile":
            # kwargs["queryset"] = ServerProfile.objects.filter(webapp_server=True)
            kwargs["queryset"] = ServerProfile.objects.filter(server_type=ServerProfile.NT)
        return super(SharedFolderAdmin, self).formfield_for_foreignkey(db_field, request, **kwargs)

    def get_queryset(self, request):
        qs = super(SharedFolderAdmin, self).get_queryset(request)
        return qs.exclude(name='Private')

    def save_related(self, request, form, *args, **kwargs):
        super(SharedFolderAdmin, self).save_related(request, form, *args, **kwargs)
        if form.instance.default:
            form.instance.users.set(list(UserProfile.objects.all()))


admin.site.register(SharedFolder, SharedFolderAdmin)


class UserProfileAdmin(admin.ModelAdmin):

    fields = ['username', 'prefix', ]
    exclude = []
    # readonly_fields = ('username',)

    original_fields = []

    def __init__(self, *args, **kwargs):
        super(UserProfileAdmin, self).__init__(*args, **kwargs)
        self.original_fields = self.fields.copy()

    def get_form(self, request, obj=None, **kwargs):
        self.exclude = []  # reset exclude list
        self.fields = self.original_fields.copy()   # restore original base fields
        filter_user_var_fields(self)                # determine appropriate UserProfile var fields and label names
        return super(UserProfileAdmin, self).get_form(request, obj, **kwargs)
    
    def save_related(self, request, form, *args, **kwargs):
        super(UserProfileAdmin, self).save_related(request, form, *args, **kwargs)

        # if a shared folder is 'default' add a new user to its list of users.
        for sfolder in SharedFolder.objects.all():
            if sfolder.default == True:
                sfolder.users.add(form.instance.id)

        # if a web server profile is the 'web app server' add this new user to the server profiles list of users.
        for sprofile in ServerProfile.objects.all():
            if sprofile.webapp_server == True:
                sprofile.users.add(form.instance.id)


admin.site.register(UserProfile, UserProfileAdmin)


###


class CustomVariableDescriptionAdminForm(forms.ModelForm):
    class Meta:
        model = VariableDescription
        fields = ['column_description', 'column_title', 'column_info',
        'column_analyzer_type', 'user_editable', 'user_viewable', 'batches', ]
        # readonly_fields = ['column_name', ]

    #def __init__(self, *args, **kwargs):
    #    super(CustomVariableDescriptionAdminForm, self).__init__(*args, **kwargs)
    #
    #   # find currently allocated column_names
    #   cols = list(VariableDescription.objects.exclude(
    #       column_name__exact='').values_list('column_name', flat=True))

    #   # acceptable ranges for char   are var{ 1,20} (only var{ 1,10} currently defined in models/db)
    #   # acceptable ranges for number are var{21,40} (only var{21,30} currently defined in models/db)
    #   # acceptable ranges for date   are var{31,60} (only var{31,50} currently defined in models/db)

    #   column_name_value = 'var1'
    #   if len(cols) > 0:
    #       # get top var num used by UserProfile e.g. var0 -> var10
    #       max_var_num = 0
    #       for field in UserProfile._meta.get_fields(include_parents=False):
    #           if field.name[0:3] == 'var':
    #               var_num = re.split(r'(\d+)', field.name[3:])
    #               if len(var_num) > 0 and len(var_num[1]) > 0:
    #                   if int(var_num[1]) > max_var_num:
    #                       max_var_num = int(var_num[1])

    #       # determine the next available UserProfile var available for this VariableDescription
    #       used_vars = set([int(n[1]) for n in [re.split(r'(\d+)', s) for s in cols]])
    #       all_vars = set(range(1, max_var_num + 1))
    #       avail_vars = list(all_vars - used_vars)
    #       avail_vars.sort()
    #       column_name_value = 'var' + str(avail_vars[0])

    #   self.fields['column_name'].initial = column_name_value
    #   self.fields['column_name'].disabled = True
    #

    def save(self, commit=True, *args, **kwargs):
        
        m = super(CustomVariableDescriptionAdminForm, self).save(commit=False, *args, **kwargs)

         # TODO detemine if it is an update or create.
        # another consideration is if an update and 
        # the previous column_analyzer_type value has changed, 
        # need to delete old value.
        previous_column_name = m.column_name
        if len(m.column_name) > 0:
            # modifying an existing entry
            VariableDescription.objects.filter(column_name=m.column_name).delete()  # remove var

        # acceptable ranges for char   are var{ 1,20} (only var{ 1,10} currently defined in models/db)
        # acceptable ranges for number are var{21,40} (only var{21,30} currently defined in models/db)
        # acceptable ranges for date   are var{41,60} (only var{41,50} currently defined in models/db)

        # for now at least, hard code floor and roof of var range for each analyzer type, not dynamically calculated.
         # find currently allocated column_names
        cols = list(VariableDescription.objects.filter(column_analyzer_type=m.column_analyzer_type).exclude(
            column_name__exact='').values_list('column_name', flat=True))

        lower = 1
        upper = 10
        if m.column_analyzer_type == 'N':
            lower = 21
            upper = 30
        elif m.column_analyzer_type == 'D':
            lower = 41
            upper = 50
        
        m.column_name = get_next_column_name(cols, lower, upper)
        m.save()
        return m


class VariableDescriptionAdmin(admin.ModelAdmin):
    form = CustomVariableDescriptionAdminForm

    actions = ['delete_model']

    def get_actions(self, request):
        actions = super(VariableDescriptionAdmin, self).get_actions(request)
        del actions['delete_selected']
        return actions

    def delete_model(self, request, obj):
        try:
            for o in obj.all():
                o.delete()
        except AttributeError:
            obj.delete()

    delete_model.short_description = 'Delete selected Variable descriptions'

    #def save_model(self, request, obj, form, change):
    #    obj.added_by = request.user
    #    super().save_model(request, obj, form, change)


admin.site.register(VariableDescription, VariableDescriptionAdmin)


class ServerProfileAdmin(admin.ModelAdmin):

    class Media:
        # css = {
        #     "all": ("my_styles.css",)
        # }
        js = ('schedule/arbutus_admin_schedule.js', )


    def save_related(self, request, form, *args, **kwargs):
        super(ServerProfileAdmin, self).save_related(request, form, *args, **kwargs)
        # if a new server profile is the 'web app' server (only one can be at any one time) 
        # add all current users to this server profile
        if form.instance.webapp_server:
            form.instance.users.set(list(UserProfile.objects.all()))


admin.site.register(ServerProfile, ServerProfileAdmin)


class ODBCProfileAdmin(admin.ModelAdmin):
     form = ODBCProfileForm


admin.site.register(ODBCProfile, ODBCProfileAdmin)


class GlobalConfigAdmin(admin.ModelAdmin):
    def has_add_permission(self, request):
        # if there's already an entry, do not allow adding
        count = GlobalConfig.objects.all().count()
        if count == 0:
            return True

        return False


admin.site.register(GlobalConfig, GlobalConfigAdmin)


def get_next_column_name(cols, lower, upper):
    column_name = 'var' + str(lower)
    if len(cols) > 0:
        used_vars = set([int(n[1]) for n in [re.split(r'(\d+)', s) for s in cols]])
        all_vars = set(range(lower, upper))
        avail_vars = list(all_vars - used_vars)
        avail_vars.sort()
        if len(avail_vars) > 0:
            column_name = 'var' + str(avail_vars[0])

    return column_name

