import struct
import itertools

from .constants import (
    CLU_CMD_OPEN,  CLU_OPENFILE, CLU_CMD_VIEW,
    REPLY_EOF, REPLY_SUCCESS, RESULT_AUTHOK,
    VIEW_GET, VIEW_SET, REPLY_FAILURE, RESULT_LOG,
)


from ..helpers import (
    get_reply, connect_socket, build_asi_packet_1, full_server_connect,
    build_asi_cmd_type_packet,
)


def build_asi_clu_cmd_open_packet(buffer):
    """

    :param buffer:
    :return:
    """
    pacsize = len(buffer)
    server = 0
    flags = 0
    data_size = len(buffer)
    dest_srce = 1
    str_fmt = '>HBBHHI%ds' % (pacsize,)
    buf = bytes(buffer)
    return struct.pack(
        str_fmt, pacsize, server, flags, CLU_CMD_OPEN,
        data_size, dest_srce, buf)


def build_asi_view_packet(buffer, dest_src):
    """

    :param buffer:
    :param dest_src:
    :return:
    """
    pacsize = len(buffer)
    server = 0
    flags = 0
    data_size = len(buffer)
    dest_srce = dest_src
    str_fmt = '>HBBHHI%ds' % (pacsize,)
    buf = bytes(buffer)
    return struct.pack(
        str_fmt, pacsize, server, flags, CLU_CMD_VIEW,
        data_size, dest_srce, buf)


def send_open_table(server_socket, table_name, table_path):
    """

    :param server_socket:
    :param table_name:
    :param table_path:
    :return:
    """
    buf = b"OPEN " + table_name.encode()
    buf += b'\000'
    buf += table_path.encode() + b'\\' + table_name.encode() + b'.fmt'
    buf += b'\000'
    send_packet = build_asi_clu_cmd_open_packet(buf)
    server_socket.send(send_packet)
    return get_reply(server_socket, CLU_OPENFILE)


def process_open_table(username, enc_password, table, take, skip, page, page_size):
    """

    :param username:
    :param enc_password:
    :param table:
    :param take:
    :param skip:
    :param page:
    :param page_size:
    :return:
    """

    with connect_socket() as server_socket:

        # prepare and send to stub
        send_packet = build_asi_packet_1(username, enc_password, True)
        server_socket.send(send_packet)
        result = get_reply(server_socket)

        columns = []
        all_data = []
        total = 0
        is_eof = False

        if result.get('cmd_type') == REPLY_SUCCESS and result.get('dest_srce') == RESULT_AUTHOK:
            full_server_connect(server_socket, username)

            # original repsonse from arbutusserver was expected and all comma (',') seperated list of field 
            # fmt metadata.  This was not always appropiate to parse here on the client side, due to a field 
            # title potentially including comma's (',') as well.
            columns_headings = prepare_columns(send_open_table(server_socket, table.name, table.path), ',')
            num_cols = len(columns_headings)

            # if no column_headings are returned lets try again with what the arbutusserver now returns,
            # which is a tab ('\t') seperated list of field fmt metadata. 
            # supported in arbutusserver build > 1938.
            if num_cols == 0:
                columns_headings = prepare_columns(send_open_table(server_socket, table.name, table.path), '\t')
                num_cols = len(columns_headings)
            
            if num_cols > 0:
                _ = send_set_view(server_socket, columns_headings, num_cols)
                get_view_data, is_eof = send_get_view(server_socket, take, skip, page, page_size)

                if skip is not None:
                    new_skip = int(skip)
                else:
                    new_skip = 0

                while True:
                    columns, data, total = generate_grid_data(
                        columns_headings,
                        get_view_data,
                        num_cols
                    )

                    all_data.extend(data)

                    # done if, not a page request
                    if page is None:
                        break
                    elif is_eof or (len(all_data) >= int(page_size)):
                        # end of arbutusserver file or the number of records
                        # equal or exceeds request page_size.
                        # regardless of what is returned from arbutus server
                        # only return page_size requested by grid.
                        sz = int(page_size)
                        if len(all_data) > sz:
                            del(all_data[int(sz):])

                        break

                    # keep asking arbutusserver for more until we reach what was asked of by the grid
                    # calculate what remains. May get slow and memory hungry with large tables.
                    if get_view_data is not None:
                        new_skip = new_skip + len(all_data) + 1

                    get_view_data, is_eof = send_get_view(server_socket, take, new_skip, page, page_size)

    # server_socket.close()
    return columns, all_data, total


def prepare_columns(format_string, separator):
    """

    :param format_string:
    :return:
    """
    # convert string data into list of dictionaries
    key_names = ['name', 'title', 'ctype', 'len']
    columns_headings = []
    column_heading = {}
    count = 1
    for (a, b) in zip(itertools.cycle(key_names), format_string.get('buffer').decode('utf-8').split(separator)):
        if a == 'len':
            b = int(b.rstrip('\000'))
        
        column_heading[a] = b
        cycled = divmod(count, len(key_names))
        if cycled[0] > 0 and cycled[1] == 0:
            columns_headings.append(column_heading)
            column_heading = {}

        count += 1

    return columns_headings


def send_set_view(server_socket, columns_headings, num_cols):
    """

    :param server_socket:
    :param columns_headings:
    :param num_cols:
    :return:
    """
    buf = str(num_cols).encode()
    buf += b'\000'  # TODO, ensure only 2 bytes for int
    for column_heading in columns_headings:
        name = column_heading.get('name', None)
        buf += name.encode()
        buf += b'\000'
        buf += b'\000'

    send_packet = build_asi_view_packet(buf, VIEW_SET)
    server_socket.send(send_packet)
    while True:
        result = get_reply(server_socket, CLU_OPENFILE)
        if result.get('cmd_type', REPLY_SUCCESS) == CLU_OPENFILE:
            send_packet = build_asi_cmd_type_packet('', REPLY_FAILURE)
            server_socket.send(send_packet)
            result = get_reply(server_socket, RESULT_LOG)

        if (result.get('cmd_type', REPLY_SUCCESS) == REPLY_SUCCESS or 
        result.get('cmd_type', REPLY_SUCCESS) == REPLY_FAILURE):
            break
        

def send_get_view(server_socket, take, skip, page, page_size):
    """

    :param server_socket:
    :param take:
    :param skip:
    :param page:
    :param page_size:
    :return:
    """

    starting_record = 1
    number_of_rows = 1
    if page_size is not None:
        if page is not None and skip is not None:
            starting_record = int(skip) + 1
            number_of_rows = int(page_size)
        else:  # TODO is this relevant
            if take == '0' and page_size == '0':
                # Excel export, get everything server-side could lag out on very large tables
                starting_record = 1
                number_of_rows = 1000000    # TODO hard coded at 1 million record limit for Excel export.

    # TODO, ensure only 2 bytes for int
    buf = str(starting_record).encode()  # "%d", starting_record) + 1;
    buf += b'\000'
    buf += b'F'  # "%s", sets[SET_HIDEFILTRECS]?"F":" ")+1;
    buf += b'\000'
    buf += str(number_of_rows).encode()  # "%d", number_of_rows) + 1;
    buf += b'\000'
    buf += str('1').encode()    # "%d", 1) + 1; // 80526100
    buf += b'\000'
    buf += str('0').encode()    # "%d", last_rec);
    buf += b'\000'
    send_packet = build_asi_view_packet(buf, VIEW_GET)
    server_socket.send(send_packet)

    # NOTE: get reply can loop twice if eof is reached
    # the first time cmd_type is CLU_FILESIZE (401)
    # and will contain number of records
    # the next recv, will have the actual data
    # we could exit first with get_reply(server_socket, CLU_FILESIZE)
    # to process this file size but it is not really required
    # for now, as well just need to know eof.
    result = get_reply(server_socket)
    is_eof = False
    if result.get('cmd_type') == REPLY_EOF:
        is_eof = True
    return result, is_eof


def generate_grid_data(columns_headings, view_data, num_cols):
    """

    :param columns_headings: 
    :param view_data: 
    :param num_cols: 
    :return: 
    """

    columns = []
    data = []

    if view_data is None:
        return columns, data, 0

    for column_heading in columns_headings:
        data_id = column_heading["name"]
        data_title = column_heading["title"].replace(';', ' ').replace('_', ' ')
        data_width = column_heading.get('len', 1)
        data_ctype = column_heading.get('ctype', '')
        columns.append({'field': data_id, 'title': data_title, 'width': data_width, 'ctype': data_ctype})

    # prepare each row of data to output as html
    view_data_str = view_data.get('buffer').decode('utf-8').split('\00\00')
    total_num_recs = 0
    view_num_recs = 0
    count = 0
    for meta in view_data_str[0].split('\x00'):
        if count == 0:
            total_num_recs = int(meta[1:].strip(' '))
        elif count == 1:
            view_num_recs = int(meta.strip(' '))
        else:   # first rec_no
            count = 0
            break

        count += 1

    recno_col = meta
    record = {}
    for col in view_data_str[1:]:
        # if count == 0:  # trim 'F' from rec. num.
        #     pass
        # handle last column, with its extra bits in buffer
        if count == (num_cols - 1):
            all_col = col.split('\x00')
            col = all_col[0]
            if len(all_col) > 1:
                recno_col = all_col[1]

        col_field = columns[count].get('field', None)
        record[col_field] = col
        count += 1
        # end of columns, add record
        if count == num_cols:
            data.append(record)
            count = 0
            record = {}

    return columns, data, total_num_recs
