from ..models import (
    Batch, SharedFolder, ServerProfile
)
from ..helpers import (
    connect_socket, get_reply, full_server_connect, get_user_prefix,
    build_asi_packet_1, build_schedule_buffer_3, build_asi_packet_sched,
    number_of_slgs_to_show
)
from .file_io import parse_batch_comment, parse_batch_file
from .user_folder import do_server_folder
from .constants import (
    REPLY_SUCCESS, RESULT_AUTHOK,
    CLU_CAN_SCHEDULE, CLU_SCHEDULE_JOB, CLU_SCHEDULE_INFO, CLU_DEL_PROFSHARED,
)


def get_user_shared_files(username, enc_password):
    """

    :param username:
    :param enc_password:
    :return:
    """

    with connect_socket() as server_socket:
        send_packet = build_asi_packet_1(username, enc_password, True)
        server_socket.send(send_packet)
        result = get_reply(server_socket)

        return_result = None, None
        if result.get('cmd_type') == REPLY_SUCCESS and result.get('dest_srce') == RESULT_AUTHOK:
            connected, error = full_server_connect(server_socket, username)
            if not connected:
                return_result = None, error
            else:
                shared_files = populate_user_shared_files(server_socket, username)
                return_result = shared_files, None

    # server_socket.close()
    return return_result


def get_user_private_files(user_profile, enc_password):
    """

    :param user_profile:
    :param enc_password:
    :return:
    """

    return_result = None
    with connect_socket() as server_socket:
        username = user_profile.username
        send_packet = build_asi_packet_1(username, enc_password, True)
        server_socket.send(send_packet)
        result = get_reply(server_socket)
        if result.get('cmd_type') == REPLY_SUCCESS and result.get('dest_srce') == RESULT_AUTHOK:
            full_server_connect(server_socket, username)
            return_result = populate_user_private_files(
                server_socket, username, get_user_prefix(user_profile)
            )

    # server_socket.close()
    return return_result


def persist_user_batches(result, user):
    """

    :param result:
    :param user:
    :return:
    """
    # delete all currently stored batches for this user
    Batch.objects.filter(user=user).delete()

    for folder in result:
        for item in folder.get('contents', None):
            batch, created = Batch.objects.update_or_create(
                tree_id=item.get('tree_id', None),
                user=user,
                name=item.get('filename', None),
                path=item.get('path', None),
                file_type=item.get('type', 'b'),  # 'b' - batch , 'd' - directory,
                shared_folder=SharedFolder.objects.get(
                    name=folder.get('shared_folder', None),
                    server_profile=ServerProfile.objects.get(webapp_server=True),
                    users=user
                )
            )

            batch.save()


def persist_user_private_files(folder, user):
    """

    :param folder:
    :param user:
    :return:
    """

    # TODO flag it is private with additional model field
    shared_folder, created = SharedFolder.objects.update_or_create(
        name='Private',
        path=get_user_prefix(user),
        server_profile=ServerProfile.objects.get(webapp_server=True)  # TODO will need fixing for multiple servers
    )
    shared_folder.save()
    shared_folder.users.add(user)

    for item in folder.get('contents', None):
        batch, created = Batch.objects.update_or_create(
            tree_id=item.get('tree_id', None),
            user=user,
            name=item.get('filename', None),
            path=item.get('path', None),
            file_type=item.get('type', 'b'),  # 'b' - batch , 'd' - directory,
            shared_folder=shared_folder
        )

        batch.save()


def populate_user_shared_files(server_socket, username):
    """

    :param server_socket:
    :param username:
    :return:
    """

    print('do_shared_folder')

    # get all shared folders
    app_server = ServerProfile.objects.get(webapp_server=True)
    user_shared_folders = SharedFolder.objects.filter(users__username__iexact=username, server_profile=app_server)  # users__username=
    entries = []
    for shared_folder in user_shared_folders:
        result = do_server_folder(server_socket, username, shared_folder.path, shared_folder.name)
        entry = {'shared_folder': shared_folder.name, 'contents': result}
        entries.append(entry)

    return entries


def populate_user_private_files(server_socket, username, folder_path):
    """

    :param server_socket:
    :param username:
    :param folder_path:
    :return:
    """

    #full_server_connect(server_socket, username)
    print('do_private_folder')
    result = do_server_folder(server_socket, username, folder_path, 'Private')
    entry = {'shared_folder': 'Private', 'contents': result}
    return entry


def process_batch_for_comment(username, enc_password, batches):
    """

    :param username:
    :param enc_password:
    :param batches:
    :return:
    """

    batch_comment = None
    with connect_socket() as server_socket:
        # prepare and send to stub
        send_packet = build_asi_packet_1(username, enc_password, True)
        server_socket.send(send_packet)
        result = get_reply(server_socket)
        if result.get('cmd_type') == REPLY_SUCCESS and result.get('dest_srce') == RESULT_AUTHOK:
            full_server_connect(server_socket, username)
            batch_comment = parse_batch_comment(server_socket, batches)

    # server_socket.close()
    return batch_comment


def process_selected_batch(username, enc_password, batches, dialog_form_data, uservar_form_data):
    """

    :param username:
    :param enc_password:
    :param batches:
    :param dialog_form_data:
    :param uservar_form_data:
    :return:
    """
    return_result = False, None
    with connect_socket() as server_socket:
        # prepare and send to stub
        send_packet = build_asi_packet_1(username, enc_password, True)
        server_socket.send(send_packet)
        result = get_reply(server_socket)
        if result.get('cmd_type') == REPLY_SUCCESS and result.get('dest_srce') == RESULT_AUTHOK:
            # TODO may have to handle multiple batches being selected
            # build batch dialog if required.
            if dialog_form_data is None:
                full_server_connect(server_socket, username)
                dialog_batch_data = parse_batch_file(server_socket, batches)
                return_result = True, dialog_batch_data
            else:
                return_result = invoke_selected_batch(
                    batches, enc_password, dialog_form_data, uservar_form_data)

    return return_result


def invoke_selected_batch(batch_info, password, dialog_form_data, uservar_form_data):
    """

    :param batch_info:
    :param password:
    :param dialog_form_data:
    :param uservar_form_data
    :return:
    """

    with connect_socket() as sched_server:
        buf = get_user_prefix(batch_info.user).encode()  # batch_info.user.prefix.encode()
        buflen = len(buf)
        send_packet = build_asi_packet_sched(CLU_CAN_SCHEDULE, buf, buflen)
        sched_server.send(send_packet)
        result = get_reply(sched_server)
        analyzer_path = None
        if result.get('cmd_type', None) == REPLY_SUCCESS:
            buffer = result.get('buffer', None)
            analyzer_path = buffer.split(b'\000')[0]

    # NOTE This step may not be required, as all it seems to do is get info on an 
    # existing procedure if it is in the job list?
    with connect_socket() as sched_server:
        buf = b''
        buf += batch_info.user.username.encode()  # sched_info->owner
        buf += b'\000'
        buf += batch_info.name.encode()
        buf += b'\000'
        buflen = len(buf)
        send_packet = build_asi_packet_sched(CLU_SCHEDULE_INFO, buf, buflen)
        sched_server.send(send_packet)
        _ = get_reply(sched_server) # a 33001 REPLY_FAILURE is acceptable.
        # potential to know here on result if something is not right but
        # no error msg in buffer until calling CLU_SCHEDULE_JOB

    # delete any residual profiles and shared txt
    with connect_socket() as sched_server:
        buf = b''
        buf += get_user_prefix(batch_info.user).encode()
        buf += '\\'.encode()
        buf += batch_info.user.username.encode()
        buf += b'\000'
        buflen = len(buf)
        send_packet = build_asi_packet_sched(CLU_DEL_PROFSHARED , buf, buflen)
        sched_server.send(send_packet)
        _ = get_reply(sched_server)

    with connect_socket() as sched_server:
        buf, buflen = build_schedule_buffer_3(
            batch_info, password, analyzer_path, dialog_form_data, uservar_form_data)
        send_packet = build_asi_packet_sched(CLU_SCHEDULE_JOB, buf, buflen)
        sched_server.send(send_packet)
        result = get_reply(sched_server)
        if result.get('cmd_type') != REPLY_SUCCESS:
            return_result = False, result.get('buffer').decode('utf-8')
        else:
            return_result = True, None

    return return_result


def get_slg_files(user_server_files):
    all_slg_files = []
    # iterate over all user files and extract *.slg files
    for afile in user_server_files:
        for acontent in afile['contents']:
            if acontent['type'] == 's':
                all_slg_files.append(acontent)

    return all_slg_files


def create_template_item(element, lidx, slg_meta):
    item = {}
    item['name'] = slg_meta['timestamp']
    item['user'] = slg_meta['user']
    item['path'] = slg_meta['path']
    item['tree_id'] = element['tree_id'] + '_' + str(lidx)
    item['file_type'] = 's'
    # additional properites, only applicable to slg's!
    item['status'] = slg_meta['status']
    item['filename'] = slg_meta['filename']
    return item


def add_slg_to_template_result(template_result, slg_meta):

    for index, elem in enumerate(template_result):
        if isinstance(elem, list):
            add_slg_to_template_result(elem, slg_meta)   
        else:
            if elem['name'] == slg_meta['batch'] and elem['file_type'].lower() == 'b':
                # guard for last element of list
                idx = index + 1
                if idx == len(template_result):
                    idx = index

                # check if array already exists, if so, add to it
                if isinstance(template_result[idx], list):
                    sub_idx = len(template_result[idx])
                    if sub_idx == number_of_slgs_to_show():
                        # remove first item on list before adding another
                        template_result[idx].pop(0)

                    template_result[idx].append(create_template_item(elem, sub_idx, slg_meta))     
                else:      
                    # else, if not, create list and add slg to it.
                    if idx == index:    # last one
                        idx += 1
                    
                    template_result.insert(idx, [create_template_item(elem, 0, slg_meta)])
               


def add_slgs_to_batches(template_result, slg_files, username):
    # disect slg filename to extract meta data.
    for file_meta in slg_files:
        try:
            slg_meta = {'filename': '', 'path': '', 'user': '', 'batch': '', 'timestamp': '', 'status': '' }
            slg_meta['filename'] = file_meta['filename']
            slg_meta['path'] = file_meta['path']
            slg_filename = file_meta['filename'].split(' ')
            slg_meta['user'] =  slg_filename[0]
            slg_meta['batch'] = slg_filename[1]
            slg_meta['timestamp'] = slg_filename[2] + ' ' + slg_filename[3].split('_')[0]
            slg_meta['status'] = slg_filename[3].split('_')[1].split('.')[0]
        except IndexError:
            continue # ignore slg, most likley log verbosity 2 generated log e.g. 'luke vendor_dallas_table se.slg'

        # use slg_meta info to find batch in template_result and add as child
        if username == slg_meta['user']:
            add_slg_to_template_result(template_result, slg_meta)
        

        
    return template_result # TODO add slg's in user server files to appropiate batches.
