import datetime
import socket
import struct

from django.db.models import Q
from django.db.models.functions import Lower

from common.constants import PASSLEN
from common.cryptor import Cryptor, encrypt_password
from common.models import Password

from .arbutus.network import get_reply
from .arbutus.constants import (
    REPLY_SUCCESS, RESULT_AUTHOK, LOGON_SUCCESS,
    CLU_CMD_SETS, CLU_CMD_COMMAND, SCHED_NOW, FILE_READ_SERVEROBJ,
    CONNECT_MSG_1, CONNECT_MSG_3, CONNECT_MSG_5, RESULT_STATUS,
    GUI_BLD, MINIMUM_COMPATIBLE, ASI_MAJOR, ASI_MINOR
)

from .models import (
    UserProfile, Batch, VariableDescription, ServerProfile, GlobalConfig, ODBCProfile
)

# TODO refactor this file into logical files to break it up
# TODO some of the arbutus specific functionality into arbutus module,
# this may require breaking some of these helper functions up, where these
# functions are just the bridge from view to arbutus specific logic.

def full_server_connect(server_socket, username):
    """
    TODO: to be diligent, this should be checked in all places it is called.
    currently only catch the successful login view when it gets user files
    which is the first time it is called.
    
    :param server_socket:
    :param username:
    :return:
    """
    # packet info from server - 'NT logon in progress, build 1206'
    print('LOGON SUCCESS')
    get_reply(server_socket, LOGON_SUCCESS)

    # connect msg 3
    print('connect msg 3')
    send_packet = build_asi_packet_3(username)
    server_socket.send(send_packet)
    get_reply(server_socket)

    # connect msg 5
    print('connect msg 5')
    send_packet = build_asi_packet_5()
    server_socket.send(send_packet)
    asipacket = get_reply(server_socket, RESULT_STATUS)
    if asipacket.get('cmd_type') == RESULT_STATUS:
        return False, asipacket.get('buffer').decode('utf-8')

    # process recv packet - Session Initialized 6.10.1206
    get_reply(server_socket, REPLY_SUCCESS)  # 0

    # initialize server
    #print('initialize server')
    #initialize_server(server_socket)
    return True, ''


def handle_connection_error(error):
    common_error_part = '  Please contact your WebConnect administrator.'
    if error is None or len(error) < 1:
        return 'Could not connect to ArbutusServer or WebConnect is not configured appropriately!' + common_error_part  # TODO error templates?
    else:
        return error + common_error_part  # TODO error template?


def build_asi_set_arrays_packet(buffer):
    """

    :param buffer:
    :return:
    """
    pacsize = 300
    server = 0
    flags = 0
    cmd_type = CLU_CMD_SETS
    data_size = 300
    dest_srce = 58
    str_fmt = '>HBBHHI300s'
    buf = bytes(buffer)
    return struct.pack(
        str_fmt, pacsize, server, flags, cmd_type,
        data_size, dest_srce, buf)


def build_asi_cmd_type_packet(buffer, cmd_type):
    """

    :param buffer:
    :param cmd_type:
    :return:
    """
    pacsize = len(buffer)
    server = 0
    flags = 0
    data_size = len(buffer)
    dest_srce = 0
    str_fmt = '>HBBHHI%ds' % (pacsize,)
    if len(buffer) == 0:
        buf = b''
    else:
        buf = bytes(buffer)

    return struct.pack(
        str_fmt, pacsize, server, flags, cmd_type,
        data_size, dest_srce, buf)


def send_vars(server_socket):
    """

    :param server_socket:
    :return:
    """
    print('------- No send/recv --------')


def send_set_arrays(server_socket):
    """

    :param server_socket:
    :return:
    """
    # TODO template of values for all users in database?
    # sets_str = [0] * 300
    sets_str = (
        1, 0, 2, 1, 0, 1, 12, 0, 0, 3, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 10, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 46, 44,
        44,
        0, 1, 0, 1, 20, 20, 32, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 50, 0, 50, 0, 50, 0, 50, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0,
        141, 68, 77, 89, 146, 40, 1, 1, 223, 0, 1, 8, 0, 0, 0, 0, 188, 0, 2, 0, 0, 230, 227, 138, 138, 202, 217, 61, 0,
        0,
        0, 5, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 10, 0, 0, 0, 0, 0, 1, 255, 3, 0, 0, 0,
        0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0,
        0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0,
        10,
        0, 0, 0, 0, 50, 25, 0)
    send_packet = build_asi_set_arrays_packet(sets_str)
    server_socket.send(send_packet)
    get_reply(server_socket)


def send_set_strings(server_socket):
    """

    :param server_socket:
    :return:
    """
    buf = b"SET DATE 'mm/dd/yyyy'"
    send_packet = build_asi_cmd_type_packet(buf, CLU_CMD_COMMAND)
    server_socket.send(send_packet)
    get_reply(server_socket)

    buf = b"SET PICTURE '-999999.99'"
    send_packet = build_asi_cmd_type_packet(buf, CLU_CMD_COMMAND)
    server_socket.send(send_packet)
    get_reply(server_socket)

    buf = b"SET DESIGNATION 'Produced with Arbutus Analyzer by: Arbutus, luke'"
    send_packet = build_asi_cmd_type_packet(buf, CLU_CMD_COMMAND)
    server_socket.send(send_packet)
    get_reply(server_socket)

    buf = b"SET PERIODS '0 31 61 91 121 151 181 10000'"
    send_packet = build_asi_cmd_type_packet(buf, CLU_CMD_COMMAND)
    server_socket.send(send_packet)
    get_reply(server_socket)

    buf = b"SET MONTHS 'Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec '"
    send_packet = build_asi_cmd_type_packet(buf, CLU_CMD_COMMAND)
    server_socket.send(send_packet)
    get_reply(server_socket)

    buf = b"SET LOOP 10000"
    send_packet = build_asi_cmd_type_packet(buf, CLU_CMD_COMMAND)
    server_socket.send(send_packet)
    get_reply(server_socket)


def initialize_server(server_socket):
    """

    :param server_socket:
    :return:
    """
    print('send_vars')
    send_vars(server_socket)
    print('send_set_arrays')
    send_set_arrays(server_socket)
    print('send_set_strings')
    send_set_strings(server_socket)


def read_sobj(server_socket, shared_folder):
    """

    :param server_socket:
    :param shared_folder:
    :return:
    """
    buf = shared_folder.encode()
    buf += b' c'
    send_packet = build_asi_cmd_type_packet(buf, FILE_READ_SERVEROBJ)
    server_socket.send(send_packet)
    get_reply(server_socket)


def connect_socket():
    """

    :return:
    """
    # send over the wire via TCP
    server_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    try:
        app_server = ServerProfile.objects.get(webapp_server=True)
        server_socket.connect((app_server.host_name, app_server.port))
    except ServerProfile.DoesNotExist:
        print("At least one server in ServerProfile must be the webapp_server")  # TODO web page warning?
        server_socket = None

    return server_socket


def build_schedule_buffer_3(batch_info, password, analyzer_path, dialog_form_data, uservar_form_data):
    """

    :param batch_info:
    :param password:
    :param analyzer_path:
    :param dialog_form_data:
    :param uservar_form_data
    :return:
    """
    raw_buf = b''
    username_bytes = batch_info.user.username.encode()  # schedule owner
    raw_buf += username_bytes
    raw_buf += b'\000'
    tot_len = len(raw_buf)

    # accommodate differences in how SQLite3 and PostreSQL store a BinaryField
    try:
        password_bytes = password.password.tobytes()    # postgreSQL
    except AttributeError:
        password_bytes = password.password              # sqlite3

    pass_len = PASSLEN + 1
    raw_buf += password_bytes[:pass_len]
    tot_len += pass_len

    raw_bufx = SCHED_NOW.to_bytes(1, 'little')  # SCHED_NOW).encode()   # frequency
    raw_bufx += b'\000'
    raw_bufx += str(0).encode()  # Number for daily & frequently
    raw_bufx += b'\000'
    # subtract 1 year approx. from now() to overcome any multi server out of sync time issues
    # as anything previous to current now() on arbutus server will be run immediately.
    ct = datetime.datetime.now() - datetime.timedelta(days=365)  # + datetime.timedelta(minutes=5)
    raw_current_time = struct.pack(
        '<HHHHHHHH',
        ct.year, ct.month,  ct.weekday(), ct.day,
        ct.hour, ct.minute, ct.second, 0)
    raw_bufx += raw_current_time
    raw_bufx += raw_current_time
    raw_bufx += b'\000\000'         # number of recurs/periodically etc.
    raw_bufx += b'\000\000'
    raw_bufx += analyzer_path       # executable path
    raw_bufx += b'\000'
    batch_name = batch_info.path.encode() + b'\\' + batch_info.name.encode()    # + b'.pro'  # procedure name/path
    if batch_info.file_type == 'b':
        batch_name += b'.pro'
    else:  
        batch_name += b'.awf'       # workflow 'B'

    raw_bufx += batch_name
    raw_bufx += b'\000'
    raw_bufx += build_variable_list_buffer(batch_info.user, dialog_form_data, uservar_form_data)
    raw_bufx += b'\000'
    raw_bufx += str(1).encode()  # result location type - (private 1, shared 2, other 3)
    raw_bufx += b'\000'
    server_profile = batch_info.shared_folder.server_profile
    prefix_bytes = get_user_prefix(batch_info.user).encode()
    raw_bufx += prefix_bytes  # .prefix.encode()  # + b'\\schedule' results go in this folder
    raw_bufx += b'\000'
    raw_bufx += prefix_bytes  # batch_info.user.prefix.encode()
    raw_bufx += b'\000'
    profile_name = server_profile.name.encode()
    raw_bufx += profile_name
    raw_bufx += b'\000'
    if delete_successful_slgs() == True:
        raw_bufx += str(1).encode()  # 0/1 Delete green logs?
    else:
        raw_bufx += str(0).encode()  # 0/1 Delete green logs?

    shared_folder_path = batch_info.shared_folder.path.encode()
    raw_bufx += shared_folder_path  # Shared Folder (Admin)
    raw_bufx += b'\000'
    raw_bufx += b'\000'  # orig_start.wYear
    raw_bufx += b'\000'  # orig_lastrun.wYear
    raw_bufx += b'\000'  # orig_lastend.wYear
    tot_len += len(raw_bufx)

    # current active needs to be first.
    profile_buf1, profile_len1 = get_server_profile(server_profile, username_bytes, prefix_bytes, password_bytes, False)
    tot_len += profile_len1

    # all server profiles info except current active
    profile_bufs = []
    for profile in ServerProfile.objects.exclude(id=server_profile.id):
        profile_buf, profile_len = get_server_profile(profile, username_bytes, prefix_bytes, password_bytes, False)
        tot_len += profile_len
        profile_bufs.append(profile_buf)

    # Add a copy of current server profile but with em_server set
    # NOTE: This is a hack, assuming that the current active server profile is also
    # an EM enabled server and capable of processing the EXCEPTION command. If the
    # arbutus server was not, the scheduled log should have the appropriate error.
    profile_buf2, profile_len2 = get_server_profile(server_profile, username_bytes, prefix_bytes, password_bytes, True)
    tot_len += profile_len2

    # shared folder
    raw_bufy = b'\000'
    raw_bufy += batch_info.shared_folder.name.encode()
    raw_bufy += b' :'
    raw_bufy += b'"' + shared_folder_path + b'"  "' + profile_name + b'"'
    raw_bufy += b'\000'
    tot_len += len(raw_bufy)

    # concatenate all part buffers
    raw_buf += raw_bufx
    raw_buf += profile_buf1
    for profile_buf in profile_bufs:
        raw_buf += profile_buf

    raw_buf += profile_buf2
    raw_buf += raw_bufy
    return raw_buf, tot_len


def build_variable_list_buffer(user, dialog_form_data, uservar_form_data):
    """

    :param user:
    :param dialog_form_data:
    :param uservar_form_data:
    :return:
    """

    # loop over user vars get values, then assign matching col. desc. to each var value
    variables = {}
    # for index in range(1, 11):
    #     var_name = 'var' + str(index)
    #     var_value = getattr(user, var_name)
    #     try:
    #         column_description = VariableDescription.objects.get(column_name=var_name).column_description.upper()
    #         variables[column_description] = var_value
    #     except ObjectDoesNotExist as dne:
    #         pass

    #all_field_types = []
    # uservar form
    if uservar_form_data:
        for field_key, field_value in uservar_form_data.items():
            lookup_key = VariableDescription.objects.get(column_name=field_key).column_description
            #TODO may need to be more resilent conversion of value to string for other types
            field_type = VariableDescription.objects.get(column_name=field_key).column_analyzer_type
            #all_field_types.append(field_type)
            variables[lookup_key.upper()] = { 'value': get_analyzer_value(field_value, field_type), 'ftype': field_type }

    # form vars - has priority if same 'key' name as user var.
    if dialog_form_data:
        for field_key, field in dialog_form_data.items():
            #all_field_types.append('C') # TODO fudge them all as 'C" types for dialog temporarily
            variables[field_key.upper()] = { 'value': field['value'], 'ftype': field['ftype'] }

    # reserved 'special' vars.
    # WHY? give the admin the ability to reference in script, without
    # having to add it as a variable description and in turn
    # customize the value for each user. An example of this
    # is the users prefix, where it is most likely the output of
    # a scripts export command will go, if they want to email that
    # file out via a NOTIFY command, they will be required to use
    # the full output path.
    # script example:
    # x=user_prefix+"\EXP002.XLSX"
    # NOTIFY SENDER "lduguid@arbutussoftware.com" MAILBOX "localhost"
    #   PORT 25 ADDRESS "%batch_address%" SUBJECT "%batch_subject%" MESSAGE "%batch_message%" ATTACHMENT "%x%"
    variables['USER_PREFIX'] = { 'value': user.prefix, 'ftype': 'C' }
    #all_field_types.append('C')
    
    # if the var has a value, add it to schedule variable buffer.
    # e.g '/vbatch_email="lduguid@arbutussoftware.com" /vbatch_subject="test email subject" '
    buf = b''
    #field_type = all_field_types.reverse()
    for var_name, var_value in variables.items():
        #field_type = all_field_types.pop()
        field_type = var_value['ftype']
        field_value = var_value['value']
        if len(field_value) > 0:
            if field_type == 'C':
                buf += b'/v' + var_name.encode() + b'="' + field_value.encode() + b'"'
            elif field_type == 'N':
                buf += b'/v' + var_name.encode() + b'=' + field_value.encode()
            elif field_type == 'L':
                buf += b'/v' + var_name.encode() + b'=' + field_value.encode()
            elif field_type == 'D':
                # convert UI date format to Arbutus date expected format, 'YYYYMMDD'
                date_sep_list = ['-', '/', '.', ' ']
                arbutus_date = ''
                for sep in date_sep_list:
                    date_list = field_value.split(sep)
                    if len(date_list) > 1: 
                        arbutus_date = date_list[0] + date_list[1] + date_list[2]
                        break
                
                # fall back
                if len(arbutus_date) < 1:
                    arbutus_date = field_value
                    
                buf += b'/v' + var_name.encode() + b'=`' + arbutus_date.encode() + b'`'
            else: 
                buf += b'/v' + var_name.encode() + b'="' + field_value.encode() + b'"'  # treat no type as 'C'

            buf += b' '

    return buf


def get_server_profile(server_profile, username_bytes, prefix_bytes, password, is_em_server):
    """

    :param server_profile:
    :param username_bytes:
    :param prefix_bytes:
    :param password:
    :param is_em_server:
    :return:
    """
    profile = b''
    server_name = ''
    if is_em_server:
        server_name = "EM_"
        
    server_name += server_profile.name
    profile += server_name.encode()         # profile name
    profile += b'\000'
    profile += str(server_profile.server_type).encode()      # server type (1 - NT)
    profile += b'\000'
    profile += username_bytes            # username
    profile += b'\000'
    tot_profile_len = len(profile)

    pass_len = PASSLEN + 1
    profile += password[:pass_len]
    tot_profile_len += pass_len

    profilex = prefix_bytes
    profilex += b'\000'
    profilex += prefix_bytes
    profilex += b'\000'
    profilex += server_profile.host_name.encode()  # server address
    profilex += b'\000'
    profilex += str(server_profile.port).encode()   # server (stub) listening port
    profilex += b'\000'
    profilex += str(server_profile.timeout).encode()  # timeout
    profilex += b'\000'
    profilex += str(server_profile.encryption).encode()  # encryption
    profilex += b'\000'
    profilex += str(server_profile.compression).encode()   # compression
    profilex += b'\000'
    # em_server
    if is_em_server:
        profilex += str(1).encode()
    else:
        profilex += str(0).encode()

    # enableSSO - hardcode to off (0) for now.
    profilex += b'\000'
    profilex += str(0).encode()

    profilex += b'\000'
    tot_profile_len += len(profilex)

    # dblist
    profileyy = b''
    for odbc_profile in ODBCProfile.objects.filter(server_profile__id=server_profile.id):
        if not is_em_server:
            profiley = b'DB::'
            profiley += odbc_profile.name.encode()                              # odbc DSN name
            profiley += b'\000'
            profiley += odbc_profile.db_username.encode()                       # database username
            profiley += b'\000'
            profiley += odbc_profile.db_password.encode()                       # database password
            profiley += b'\000'
            tot_profile_len += len(profiley)
            # add current DB profile to DB profile list
            profileyy += profiley

    # concat partial buffer parts.
    profilex += profileyy   # add DB list to current profile.
    profile += profilex
    return profile, tot_profile_len


def arbutus_wc_authenticate(username, password):
    """
    Public helper to authenticate with ASI stub/server
    :param username:
    :param password:
    :return:
    """

    server_socket = None

    try:
        server_socket = connect_socket()
        send_packet = build_asi_packet_1(username, password, False)
        server_socket.send(send_packet)
        result = get_reply(server_socket)
    except ConnectionRefusedError:
        server_socket.close()
        return False

    if result.get('cmd_type') == REPLY_SUCCESS and result.get('dest_srce') == RESULT_AUTHOK:
        server_socket.close()
        return True
    else:
        server_socket.close()
        return False


def get_enc_password(username):
    """

    :param username:
    :return:
    """
    return Password.objects.filter(username__iexact=username).first()


def build_connect_msg_1(username, password, is_encrypted):
    """

    :param username:
    :param password:
    :param is_encrypted:
    :return:
    """
    buf_raw = str(0).encode()  # sets[290]=0000
    buf_raw += b'\000'
    buf_raw += b'WEBCONNECT_CLIENT/'  # client name #
    buf_raw += username.encode()      # user name logged in
    buf_raw += b'/W'                  # client type - /W = webconnect
    buf_raw += b'\000'
    buf_raw += username.encode()      # b'luke_duguid' 	# User id
    buf_raw += b'\000'
    tot_len = len(buf_raw)
    if is_encrypted:
        # accommodate differences in how SQLite3 and PostreSQL store a BinaryField
        try:
            enc_passwd = password.password.tobytes()    # postgreSQL
        except AttributeError:
            enc_passwd = password.password              # SQLite3
    else:
        enc_passwd = encrypt_password(password)

    pass_len = PASSLEN + 1
    buf_raw += enc_passwd[:pass_len]
    tot_len += pass_len
    buf_rawx = str(1).encode()      # PROFILE_ENCRYPT - password HAS to be encrypted buffer!
    buf_rawx += b'\000'
    buf_rawx += b'IMS:OFF'          # PROFILE_IMS   TODO
    buf_rawx += b'\000'
    buf_rawx += b'SET281:OFF'       # SETS281   TODO
    buf_rawx += b'\000'
    tot_len += len(buf_rawx)
    buf_raw += buf_rawx
    return buf_raw, tot_len


def build_asi_packet_1(username, password, is_encrypted):
    """

    :param username:
    :param password:
    :param is_encrypted:
    :return:
    """

    with Cryptor() as cryptor:
        cryptor.resetkey()
        buffer, bufferlen = build_connect_msg_1(username, password, is_encrypted)
        result, enc_buffer, enc_buflen = cryptor.encrypt_buf(buffer, bufferlen)

    pacsize = enc_buflen.value              # socket.htons(buflen.value)
    server = 0
    flags = 0
    cmd_type = CONNECT_MSG_1                # socket.htons(CONNECT_MSG_1)
    data_size = enc_buflen.value            # socket.htons(buflen.value)
    dest_srce = MINIMUM_COMPATIBLE          # socket.htonl(MINIMUM_COMPATIBLE)
    str_fmt = '>HBBHHI%ds' % (enc_buflen.value, )
    return struct.pack(
        str_fmt, pacsize, server, flags, cmd_type,
        data_size, dest_srce, enc_buffer.raw)


def build_connect_msg_3(username):
    """

    :param username:
    :return:
    """

    user = UserProfile.objects.get(username__iexact=username)
    app_server = ServerProfile.objects.get(webapp_server=True)

    tot_len = 0
    buf_raw = b'\0\0'                    # compression level
    buf_raw += b'\0\0'                   # encryption
    buf_raw += b'\0'
    buf_raw += str(4).encode()           # servier_id (profile ?) TODO
    buf_raw += b'\000'
    buf_raw += get_user_prefix(user).encode()
    buf_raw += b'\000'
    buf_raw += app_server.name.encode()  # server name
    buf_raw += b'\000'
    buf_raw += str(3600).encode()        # timeout
    buf_raw += b'\000'
    buf_raw += str(MINIMUM_COMPATIBLE).encode()
    buf_raw += b'\000'
    buf_raw += str(GUI_BLD).encode()
    buf_raw += b'\000'
    buf_raw += str(ASI_MAJOR).encode()
    buf_raw += b'\000'
    buf_raw += str(ASI_MINOR).encode()
    buf_raw += b'\000'
    buf_raw += b'0'                      # scheduled_job
    buf_raw += b'\000'
    tot_len += len(buf_raw)
    return buf_raw, tot_len


def build_asi_packet_3(username):
    """

    :return:
    """
    buffer, buflen = build_connect_msg_3(username)
    pacsize = buflen
    server = 0
    flags = 0
    cmd_type = CONNECT_MSG_3
    data_size = buflen
    dest_srce = 0
    str_fmt = '>HBBHHI%ds' % (buflen, )
    return struct.pack(
        str_fmt, pacsize, server, flags, cmd_type,
        data_size, dest_srce, buffer)


def build_asi_packet_5():
    """

    :return:
    """
    pacsize = 0
    server = 0
    flags = 0
    cmd_type = CONNECT_MSG_5
    data_size = 0
    dest_srce = 0
    return struct.pack(
        '>HBBHHI', pacsize, server, flags, cmd_type, data_size, dest_srce)


def build_asi_packet_sched(cmd_type, buffer, buflen):
    """

    :param cmd_type:
    :param buffer:
    :param buflen:
    :return:
    """
    pacsize = buflen
    server = 0
    flags = 0
    cmd_type = cmd_type
    data_size = buflen
    dest_srce = MINIMUM_COMPATIBLE
    str_fmt = '>HBBHHI%ds' % buflen
    return struct.pack(
        str_fmt, pacsize, server, flags, cmd_type,
        data_size, dest_srce, buffer)


def prepare_batches_for_template(user, entries):
    """
    converts 'flat' query list of all shared folders and files from db
    and makes it a nested list (parent/child) so that it can be consumed
    by a modified (add jstree attributes to tags) unordered_list template tag.
    :param user:
    :param entries:
    :return:
    """

    current_list = []
   
    for entry in entries.order_by(Lower('name')):  # consistent shared/private folder files order.
        # append node to current list
        node = {'name': entry.name, 'path': entry.path, 'tree_id': entry.tree_id, 'file_type': entry.file_type}
        current_list.append(node)
        if entry.file_type == 'd':    # this is a directory, gets its children until no more
            # get children of current directory
            child_dirs = Batch.objects.filter(
                user=user,
                path=entry.path+'\\'+entry.name,
            ).exclude(
                Q(file_type='f') & ~Q(path__startswith=entry.path+'\\'+entry.name)
            ).exclude(
                Q(file_type='s')
            )

            if len(child_dirs) == 0:
                continue

            # we have children, create a new list and append it to the current list
            current_list.append(prepare_batches_for_template(user, child_dirs))

    return current_list


# Global Config accessor functions

def get_user_prefix(user):
    """

    :param user:
    :return:
    """
    user_prefix = ''

    # if user has prefix, use it
    if len(user.prefix) > 0:
        user_prefix = user.prefix
    else:
        # if user.prefix is blank, fallback to GlobalConfig
        try:
            gc = GlobalConfig.objects.get()  # should only ever be one
            if len(gc.user_prepended_prefix) > 0:
                # only set if not blank
                user_prefix = gc.user_prepended_prefix + '\\' + user.username
        except GlobalConfig.DoesNotExist:
            pass    # no globalconfig object defined

    return user_prefix


def get_logo():
    logo = None

    try:
        gc = GlobalConfig.objects.get()  # should only ever be one
        logo = gc.logo
    except GlobalConfig.DoesNotExist:
        pass

    return logo


def show_data_tables():
    result = False

    try:
        gc = GlobalConfig.objects.get()
        result = gc.show_data_tables
    except GlobalConfig.DoesNotExist:
        pass

    return result


def show_private_folders():
    result = False

    try:
        gc = GlobalConfig.objects.get()
        result = gc.show_private_folders
    except GlobalConfig.DoesNotExist:
        pass

    return result

def delete_successful_slgs():
    result = False

    try:
        gc = GlobalConfig.objects.get()
        result = gc.delete_successful_slgs
    except GlobalConfig.DoesNotExist:
        pass

    return result

def number_of_slgs_to_show():
    result = 5

    try:
        gc = GlobalConfig.objects.get()
        result = gc.number_of_slgs_to_show
    except GlobalConfig.DoesNotExist:
        pass

    return result

def get_partial_template(user_batch_var):
    # the 'convention for naming the partial templates
    # are 'user_var_' + {type} + '_' + {'editable.html'||'viewable.html'}
    # defaults to no {type}, which is a standard edit input
    base_template_name = 'schedule/partial/'
    template_name = base_template_name + 'user_var_'

    if user_batch_var['user_var_type'] == 'N':
        template_name += 'number_'
    elif user_batch_var['user_var_type'] == 'D':
         template_name += 'date_'

    # is this editable or viewable 
    template_name += 'editable.html'
    if not user_batch_var['user_editable']:
        template_name += 'viewable.html'

    return template_name


def get_value_by_type(value, type):
    # convert datetime object to appropiate string format for widget consuption.
    return_value = value
    if type == 'D':
        return_value = value.isoformat()

    return return_value

def get_analyzer_value(value, type):
    # convert internal format to analyzer expected format.
    return_value = value
    if type == 'D':
        dt_values = value.split('-')
        if len(dt_values) == 3:
            year = dt_values[0]
            month = dt_values[1]
            day = dt_values[2]
            return_value = year + month + day

    return return_value