import re
from datetime import date

from django.db import models
from django.forms import forms

from model_utils.models import TimeStampedModel

from .fields import OneTrueBooleanField

# user centric data that can be edited by an admin level user


class UserProfile(TimeStampedModel):
    username = models.CharField(max_length=128, unique=True)
    prefix = models.CharField(max_length=1024, blank=True, help_text="Users private folder on Arbutus hub server.")

    # generic variable values particular to this user
    # the column name it matches is defined in 'VariableDescription'

    # reserved Char fields
    var1 = models.CharField(max_length=256, blank=True)
    var2 = models.CharField(max_length=256, blank=True)
    var3 = models.CharField(max_length=256, blank=True)
    var4 = models.CharField(max_length=256, blank=True)
    var5 = models.CharField(max_length=256, blank=True)
    var6 = models.CharField(max_length=256, blank=True)
    var7 = models.CharField(max_length=256, blank=True)
    var8 = models.CharField(max_length=256, blank=True)
    var9 = models.CharField(max_length=256, blank=True)
    var10 = models.CharField(max_length=256, blank=True)

    # reserved Number fields, using Decimal, not sure if it is appropiate
    var21 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var22 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var23 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var24 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var25 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var26 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var27 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var28 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var29 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)
    var30 = models.DecimalField(max_digits=19, decimal_places=10, default=0.0)

    #reserved Date Fields
    var41 = models.DateField(default=date.today)
    var42 = models.DateField(default=date.today)
    var43 = models.DateField(default=date.today)
    var44 = models.DateField(default=date.today)
    var45 = models.DateField(default=date.today)
    var46 = models.DateField(default=date.today)
    var47 = models.DateField(default=date.today)
    var48 = models.DateField(default=date.today)
    var49 = models.DateField(default=date.today)
    var50 = models.DateField(default=date.today)

    # TODO other user profile state could go here

    def __str__(self):
        return self.username

    # TODO should we put server profile info like in and out prefix as user specific?
    # TODO to extend on this idea, lets have general common, shared folder and server profiles
    # not associated with a userprofile, and then use and intermediary table (many-to-many?)
    # to 'associate' these with each user.

    # we can also have a default column added to both (only one server) and shared folders
    # so that a userprofile will automatically be associated with these defaults on there
    # first successful login.


class ServerProfile(TimeStampedModel):
    MVS = 0
    NT = 1
    NIX = 2
    AS400 = 3
    LOCAL = 4
    SERVER_TYPE_CHOICES = (
        (MVS, 'MVS'),
        (NT, 'Windows'),
        (NIX, 'Linux'),
        (AS400, 'AS400'),
        (LOCAL, 'Local'),
    )
    users = models.ManyToManyField(UserProfile, help_text="All users that will use this Arbutus hub server profile.")
    server_type = models.IntegerField(choices=SERVER_TYPE_CHOICES, default=NT, help_text='Arbutus hub server platform type')
    name = models.CharField(max_length=256, help_text='Name of Arbutus hub server')
    host_name = models.CharField(max_length=256, help_text='Network hostname or IP')
    port = models.IntegerField(help_text='Port Arbutus hub server is listening on')
    timeout = models.IntegerField(default=60)
    encryption = models.IntegerField(default=0)
    compression = models.IntegerField(default=0)

    webapp_server = OneTrueBooleanField(default=False, help_text='Arbutus hub server used by WebConnect.')

    def __str__(self):
        return self.name


class ODBCProfile(TimeStampedModel):
    name = models.CharField(max_length=128, verbose_name="DSN Name", help_text="DSN name as labeled in ODBC manager")
    db_username = models.CharField(max_length=128, unique=True, verbose_name="Database username",  help_text="Database username for ODBC DSN.")
    # NOTE: We never store the DB password as raw text, it is encrypted with ASI encrytpion first, then hexified for persistence in a CharField
    # We use BinaryField for authentication as it is easier then hexifying and de-hexifying when it is required, DB passwords are just
    # sent to server as hexified format so it makes sense to store it as such.
    db_password = models.CharField(max_length=128, verbose_name="Database password", help_text="Database password for ODBC DSN.")
    server_profile = models.ForeignKey('ServerProfile', on_delete=models.CASCADE, help_text='Arbutus Hub server location for this ODBC DSN profile.')

    def __str__(self):
        return self.name


class SharedFolder(TimeStampedModel):
    users = models.ManyToManyField(UserProfile, help_text='All users that have access to this shared folder.')
    name = models.CharField(max_length=256, help_text='Shared folder name')
    path = models.CharField(max_length=1024, help_text='Folder location of shared folder on Arbutus hub server')
    server_profile = models.ForeignKey('ServerProfile', on_delete=models.CASCADE, help_text='Arbutus hub server location of this shared folder')
    project_name = models.CharField(max_length=256, blank=True)
    default = models.BooleanField(default=False, help_text='Common shared folder for ALL users.')

    def __str__(self):
        return self.name

    # TODO should SharedProfile be Many2Many with UserProfile? Should we create more basic
    # template tables for serverprofile and sharedfolder shared across all users
    # and then add user specific details to these ones
    # i.e. hostname and port for a particular server name
    # would be common to all users, whereas in and out prefix would be
    # user specific. Also all users on first successful log in
    # should probably be assigned a default server and shared folder
    # that everyone has access to or else currently, they could
    # log into the system but not be assigned anything, which in turn
    # would break the code, we could also handle this situation and give
    # the user a warning that the admin is required to do it, however,
    # it could make the admins job easier to assign some defaults.


class VariableDescription(TimeStampedModel):
    # example:       |column_name|column_description|
    #   `            --------------------------------
    #                |var1       |batch_email       |
    #                |var2       |name
    #                |...
    #
    # Note that the var descriptions would be common for all
    # users of each web app installation.
    # 11/1/2021 - adding column type - this is a mapping to analyzer types.
    
    CHAR = 'C'
    NUMBER = 'N'
    DATE = 'D'
    ANALYZER_TYPES = [
        (CHAR,   'CHAR'),
        (NUMBER, 'NUMBER'),
        (DATE,   'DATE'),
    ]

    column_name = models.CharField(max_length=64)
    column_description = models.CharField(
        max_length=128,  verbose_name="Variable Name", help_text='Must match variable name as referenced in procedure')
    column_title = models.CharField(
        max_length=128, verbose_name="Variable Title", help_text='Descriptive label for variable referenced on the procedure form')
    column_info =  models.CharField(
        max_length=256, blank=True, verbose_name="Variable Description",
        help_text='Variable hint to best describe the variables format and purpose.')

    column_analyzer_type = models.CharField(max_length=2,
        verbose_name="Variable Type",
        choices=ANALYZER_TYPES, 
        default=CHAR, help_text='Analyzer field type that best describes what the procedure expects.')

    user_editable = models.BooleanField(default=True, help_text='User can edit the value of this variable')
    user_viewable = models.BooleanField(default=True, help_text='User can view the value of this variable')
    batches = models.CharField(
        max_length=4096, blank=True,
        verbose_name="Procedures",
        help_text='Comma separated list of procedure names that will use this variable. If empty, used by all.')

    def __str__(self):
        return self.column_description

    def delete(self):
        """
        blank out var[1-10] value if the corresponding variable
        description has been deleted.
        :return:
        """
        for user_profile in UserProfile.objects.all():
            # clear/default the deleted variable description
            if self.column_analyzer_type == 'N':
                setattr(user_profile, self.column_name, 0)
            elif self.column_analyzer_type == 'D':
                setattr(user_profile, self.column_name, date.today())     
            else:
                setattr(user_profile, self.column_name, '')     

            user_profile.save()

        super(VariableDescription, self).delete()


class GlobalConfig(TimeStampedModel):

    ONE = 1
    FIVE = 5
    TEN = 10
    TWENTYFIVE = 25
    ONEHUNDRED = 100
    SLG_NUMBER_SHOWN_CHOICES = [
        (ONE, '1'),
        (FIVE, '5'),
        (TEN, '10'),
        (TWENTYFIVE, '25'),
        (ONEHUNDRED, '100'),
    ]

    user_prepended_prefix = models.CharField(
        max_length=1024, blank='True',
        help_text='Folder location on Arbutus hub server that can contain all users private folders')
    logo = models.ImageField(blank=True, help_text="Company logo")
    show_data_tables = models.BooleanField(default=True, help_text='Show data tables in user shared folders.')
    show_private_folders = models.BooleanField(
        default=True, help_text='Show users private folders with shared folders.')
    delete_successful_slgs = models.BooleanField(
        default=True, help_text='Does not create a log file for successful procedures.')
    number_of_slgs_to_show = models.IntegerField(
        choices=SLG_NUMBER_SHOWN_CHOICES,
        default=FIVE,
        help_text='Show number of most recent procedure log files to display.')
    # active = OneTrueBooleanField(default=False)

    def __str__(self):
        return 'WebConnect global configuration'


# working tables to persist state across view request/response or sessions etc.
# Should probably _NOT_ be exposed to admin

class Batch(TimeStampedModel):
    # unique_together = ("tree_id", "user")
    tree_id = models.CharField(max_length=256)
    user = models.ForeignKey('UserProfile', on_delete=models.CASCADE)
    shared_folder = models.ForeignKey('SharedFolder', on_delete=models.CASCADE)
    path = models.CharField(max_length=1024)
    name = models.CharField(max_length=256)
    file_type = models.CharField(max_length=9)


class UIConfig(TimeStampedModel):
    # NOTE: why not us a Batch foreign key? Batch table
    # is cleaned on every page refresh which will
    # causes cascading deletes not to mention all
    # id's/tree_id's in batch could be invalid after
    # a refresh etc.
    # Is path/name/file_type unique enough?
    user = models.ForeignKey('UserProfile', on_delete=models.CASCADE)
    path = models.CharField(max_length=1024)
    name = models.CharField(max_length=256)
    file_type = models.CharField(max_length=9)
    grid_options = models.TextField()


# model helpers

def filter_user_var_fields(instance):
    """

    :param instance:
    :return:
    """
    for f in UserProfile._meta.get_fields():
        field_name = f.__getattribute__('name')
        if re.search(r'var\d+$', field_name):
            try:
                UserProfile._meta.get_field(field_name).verbose_name = (
                    VariableDescription.objects.get(column_name=field_name)
                )
                instance.fields.append(field_name)
            except VariableDescription.DoesNotExist:
                instance.exclude.append(field_name)
