import json
from collections import OrderedDict

from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.models import User
from django.shortcuts import render, HttpResponse, HttpResponseRedirect
from django.views.generic import FormView
from django.template.loader import render_to_string

from django.contrib.auth.mixins import LoginRequiredMixin
from django.shortcuts import redirect

from common.auth import save_arbutus_passwd

from .helpers import (
    get_enc_password, prepare_batches_for_template,
    get_user_prefix, get_logo, show_private_folders,
    handle_connection_error, get_partial_template, get_value_by_type,
)
from .auth import save_user_profile
from .forms import LoginForm, SuccessForm
from .models import (
    UserProfile, ServerProfile, SharedFolder, VariableDescription,
    Batch, UIConfig
)
from .arbutus.user_files import (
    get_user_shared_files, process_selected_batch, persist_user_batches,
    process_batch_for_comment, get_user_private_files, persist_user_private_files,
    get_slg_files, add_slgs_to_batches,
)
from .arbutus.table_grid import process_open_table


class LoginView(FormView):
    template_name = 'schedule/index.html'
    form_class = LoginForm
    success_url = 'session/'

    def get(self, request, *args, **kwargs):
        form = self.form_class()

        return render(
            request,
            self.template_name,
            {'form': form, 'logo':  get_logo(), }
        )

    def form_valid(self, form):

        # NOTE: Windows logon has a case insensitive username. 
        # Attention is required to prevent accidental creation of 
        # multiple user profiles for the same user when they use 
        # different username case and are still able to successfully authenticate.
        username = form.cleaned_data['username']
        password = form.cleaned_data['password']

        # remove any existing windows login user from User table before auth.
        try:
            # if user exists and is not a admin user, use a case-insensitive comparison of username.
            User.objects.get(username__iexact=username, is_staff=False, is_superuser=False).delete()
        except User.DoesNotExist:
            pass

        #logout(self.request)
        user = authenticate(
            self.request,
            username=username,
            password=password,
            #arbutus_authenticate=arbutus_wc_authenticate
        )

        # do not reserved user 'admin' as it is not a windows login authenticated user
        if user is not None and user.is_active and user.is_superuser != True:
            save_user_profile(username)
            save_arbutus_passwd(username, password)
            logout(self.request)
            login(self.request, user)
            return super(LoginView, self).form_valid(form)
        else:
            form.add_error('username', '')
            form.add_error('password', 'Credentials incorrect! Try again.')
            return self.form_invalid(form)


class SuccessView(LoginRequiredMixin, FormView):
    template_name = 'schedule/work.html'
    form_class = SuccessForm
    success_url = '/webconnect/'  # success is logout
    #
    login_url = '/webconnect/'
    redirect_field_name = 'redirect_to'

    # initial = {'username': 'value'}
    def get(self, request, *args, **kwargs):
        form = self.form_class()  # initial=self.initial)
        context_data = OrderedDict()
        if request.user.is_authenticated:
            try:
                user_profile = UserProfile.objects.get(username__iexact=request.user.username)
                if len(get_user_prefix(user_profile)) < 1:
                    return HttpResponse(handle_connection_error(
                        'You currently do not have a private folder prefix set.  '
                        'Please contact you WebConnect administrator.'))
                try:
                    # start with a clean slate with the batches table and private share folders.
                    Batch.objects.filter(user=user_profile).delete()
                    private_folder = SharedFolder.objects.get(users=user_profile, name='Private')
                    # remove relationships in intermediate many2many table
                    private_folder.users.remove(user_profile)
                    user_profile.sharedfolder_set.remove(private_folder)
                    # as the private folder is a pseudo shared folders, also remove it from shared folders
                    private_folder.delete()
                except SharedFolder.DoesNotExist:
                    pass

                user_server_files, error = get_user_shared_files(
                    request.user.username,
                    get_enc_password(request.user.username)
                )
                if user_server_files is None:
                    return HttpResponse(handle_connection_error(error))

                persist_user_batches(user_server_files, user_profile)

                # do not bother with private folder and contents unless a
                # user has a prefix and global config is showing private folders
                if len(get_user_prefix(user_profile)) > 0 and show_private_folders():
                    user_private_files = get_user_private_files(
                        user_profile,
                        get_enc_password(request.user.username)
                    )
                    persist_user_private_files(
                        user_private_files,
                        user_profile
                    )
                    user_server_files.append(user_private_files)

                all_slg_files = get_slg_files(user_server_files)
                #persist_user_slg_files(
                #    user_slg_files,
                #    user_profile
                #)
                
                # consistent shared/private folder order.
                user_server_files.sort(key=lambda x: x['shared_folder'], reverse=False)
                
                for folder in user_server_files:
                    folder_name = folder.get('shared_folder', None)
                    app_server = ServerProfile.objects.get(webapp_server=True)
                    shared_folder = SharedFolder.objects.get(
                        name=folder_name,
                        server_profile=app_server,
                        users=user_profile
                    )
                    batches = Batch.objects.filter(user=user_profile,
                                                   shared_folder=shared_folder,
                                                   path=shared_folder.path)
                    template_result = prepare_batches_for_template(user_profile, batches)
                    slg_added_result = add_slgs_to_batches(template_result, all_slg_files, user_profile.username)
                    context_data[folder_name] = slg_added_result

            except UserProfile.DoesNotExist:
                return HttpResponseRedirect('/admin/')  # Redirect to admin if valid auth but UserProfile does not exist

            return render(
                request,
                self.template_name,
                {'form': form, 'result': context_data, 'logo': get_logo(), 'user_name': request.user.username,}
            )

    def form_valid(self, form):
        logout(self.request)
        return super(SuccessView, self).form_valid(form)

class LogoutView(LoginRequiredMixin, FormView):

    def get(self, request, *args, **kwargs):
        logout(request)
        return redirect('login')

class ProcessView(SuccessView):

    # user has clicked 'run' button, parse batch and get dialog variable values if any
    def get(self, request, *args, **kwargs):
        # form = self.form_class()  # initial=self.initial)
        error_msg = None
        if request.user.is_authenticated:
            user = UserProfile.objects.get(username__iexact=request.user.username)
            if len(get_user_prefix(user)) == 0:
                error_msg = 'Unable to run, no user prefix defined. Please contact WebConnect admin'
            batch_info = Batch.objects.get(
                tree_id=request.GET.get(key='batch_id'),
                user=user
            )
            # TODO get which button type,i.e. run or dialog and make decision accordingly.
            result, dialog_data = process_selected_batch(
                request.user.username,
                get_enc_password(request.user.username),
                batch_info,
                None,
                None
            )

        # return render(request, self.template_name, {'form': form, 'dialog_data': dialog_data, }, )
        return HttpResponse(json.dumps(
            {'dialog_data': dialog_data, 'app_error': error_msg}),
            content_type='application/json'
        )

    # invoke batch using dialog values if any, else just user variables are used.
    def post(self, request, *args, **kwargs):
        error_msg = None
        if request.user.is_authenticated:
            batch_info = Batch.objects.get(
                tree_id=request.POST.get(key='batch_id'),
                user=UserProfile.objects.get(username__iexact=request.user.username)
            )
            dialog_form_data = json.loads(request.POST.get(key='dialog_form_data'))
            uservar_form_data = json.loads(request.POST.get(key='uservar_form_data'))
            result, process_data = process_selected_batch(
                request.user.username,
                get_enc_password(request.user.username),
                batch_info,
                dialog_form_data,
                uservar_form_data
            )

            if result is False:
                error_msg = process_data
                process_data = None

        return HttpResponse(json.dumps(
            {'process_data': process_data, 'app_error': error_msg}),
            content_type='application/json'
        )


class RenderBatchView(LoginRequiredMixin, FormView):

    def get(self, request, *args, **kwargs):
        html = '<form id="user_vars">'
        if request.user.is_authenticated:
            batch_id = request.GET.get(key='batch_id')
            if request.headers.get('x-requested-with') == 'XMLHttpRequest':
                for user_batch_var in self.all_user_batch_vars(request.user.username, batch_id):
                    template_name = get_partial_template(user_batch_var)
                    html += render_to_string(
                        template_name,
                        {'user_var_id': user_batch_var['user_var_id'],
                          'user_var_label': user_batch_var['user_var_label'],
                          'user_var_value': user_batch_var['user_var_value'],
                          'user_var_info': user_batch_var['user_var_info']}
                    )

                # look for comments in batch
                batch_info = Batch.objects.get(
                    tree_id=request.GET.get(key='batch_id'),
                    user=UserProfile.objects.get(username__iexact=request.user.username)
                )
                html += render_to_string(
                    'schedule/partial/batch_comment.html',
                    {'batch_comment': process_batch_for_comment(
                        request.user.username,
                        get_enc_password(request.user.username),
                        batch_info)})

        html += '</form>'
        return HttpResponse(html)

    def all_user_batch_vars(self, username, batch_id):
        html_vars = []
        for var in VariableDescription.objects.all():
            # check if optional batches are listed.
            if var.batches == '' or Batch.objects.get(
                    tree_id=batch_id,
                    user=UserProfile.objects.get(username__iexact=username)
            ).name.upper() in set([x.strip().upper() for x in var.batches.split(',')]):
                # TODO cannot be NOT viewable BUT editable. Best to constrain in admin
                if var.user_editable or var.user_viewable:
                    html_vars.append(self.create_html_var(var, username))

        return html_vars

    @staticmethod
    def create_html_var(var, username):
        # user column title if there is one, else use column description.
        if len(var.column_title) < 1:
            user_var_label = var.column_description
        else:
            user_var_label = var.column_title
        
        return {'user_var_id': var.column_name,
                'user_var_label': user_var_label,
                'user_var_value': get_value_by_type(
                    getattr(UserProfile.objects.get(username__iexact=username), var.column_name), var.column_analyzer_type ),
                'user_var_info': var.column_info,
                'user_var_type': var.column_analyzer_type,
                'user_viewable': var.user_viewable,
                'user_editable': var.user_editable}


class GridTableDataView(LoginRequiredMixin, FormView):

    def get(self, request, *args, **kwargs):

        columns = []
        data = []
        total = 0

        if request.user.is_authenticated:
            if request.headers.get('x-requested-with') == 'XMLHttpRequest':
                table_id = request.GET.get(key='table_id')
                # on grid datasource transport reads, additional grid variables
                take = request.GET.get(key='take')
                skip = request.GET.get(key='skip')
                page = request.GET.get(key='page')
                page_size = request.GET.get(key='pageSize')

                table = Batch.objects.get(
                    tree_id=table_id,
                    user=UserProfile.objects.get(username__iexact=request.user.username)
                )

                # open table and read column information (get_it_yourself)
                columns, data, total = process_open_table(
                    request.user.username,
                    get_enc_password(request.user.username),
                    table,
                    take,
                    skip,
                    page,
                    page_size
                )

        #if page is None:
        #   return HttpResponse(json.dumps({'columns': columns, 'data': data }))
        #else:
            # if final is less than requested page size
            #final_page_size = page_size
            #if int(total) - int(skip) < int(page_size):
            #    final_page_size = int(total) - int(skip)

        try:
            user = UserProfile.objects.get(username__iexact=request.user.username)
            batch = Batch.objects.get(
                tree_id=table_id,
                user=user
            )
            ui_config = UIConfig.objects.get(
                user=user,
                name=batch.name,
                path=batch.path,
                file_type='f'
            )
            grid_config_options = ui_config.grid_options
        except UIConfig.DoesNotExist:
            grid_config_options = None

        return HttpResponse(
            json.dumps(
                {'columns': columns,
                 'data': data,
                 'total': total,
                 'grid_options': grid_config_options}
            ),
            content_type='application/json'
        )

    def post(self, request, *args, **kwargs):

        table_id = request.POST.get('table_id', None)
        grid_column_options = request.POST.get('grid_options', None)

        try:
            user = UserProfile.objects.get(username__iexact=request.user.username)
            batch = Batch.objects.get(
                tree_id=table_id,
                user=user
            )
            obj, created = UIConfig.objects.update_or_create(
                user=user,
                name=batch.name,
                path=batch.path,
                file_type='f'
            )
            obj.grid_options = grid_column_options
            obj.save()
            saved_ok = True
        except Exception as e:
            saved_ok = False

        return HttpResponse(
            json.dumps({'saved_ok': saved_ok}), content_type='application/json'
        )
